package spec

import (
	"strconv"
	"strings"
	"time"

	"github.com/pkg/errors"
)

// Chassis represents a Kontron Device
type Chassis struct {
	ID             int       `db:"id" json:"-"`
	Component      string    `db:"component" json:"component"`
	Number         int       `json:"-"`
	SerialNumber   int       `db:"serial_number" json:"serial_number"`
	Pop            string    `db:"pop" json:"pop"`
	Rack           string    `db:"rack" json:"rack"`
	RackUnit       int       `db:"ru" json:"rack_unit"`
	CommitDateTime time.Time `db:"commit_datetime" json:"commit_datetime"`
	CommitHash     string    `db:"buildsheet_md5" json:"buildsheet_md5"`
	UpdatedAt      time.Time `db:"db_last_updated" json:"-"`

	HubNodes []*HubNode `json:"hubnodes"`
	Nodes    []*Node    `json:"nodes"`
}

func newChassis(attrs map[string]string, pop, rack string, rackUnit int, commitTime time.Time, commitHash string) (*Chassis, error) {
	var (
		ch         Chassis
		unexpected map[string]string
	)

	for attr, val := range attrs {
		switch attr {
		case "Component":
			ch.Component = val
		case "Number":
			num, err := strconv.Atoi(val)
			if err != nil {
				return nil, errors.Errorf("Failed to parse chassis number as integer: %v", val)
			}
			ch.Number = num
		case "SerialNumber":
			num, err := strconv.Atoi(val)
			if err != nil {
				return nil, errors.Errorf("Failed to parse serial number as integer: %v", val)
			}
			ch.SerialNumber = num
		default:
			if unexpected == nil {
				unexpected = make(map[string]string)
			}
			unexpected[attr] = val
		}
	}

	if len(unexpected) > 0 {
		return nil, errors.Errorf("Unexpected attributes parsed for chassis: %v", unexpected)
	}

	ch.Pop, ch.Rack, ch.RackUnit, ch.CommitDateTime, ch.CommitHash = pop, rack, rackUnit, commitTime, commitHash

	if err := ch.Validate(); err != nil {
		return nil, err
	}

	return &ch, nil
}

// Validate ensures we have no unexpected initialized values for a Chassis object
func (ch *Chassis) Validate() error {
	var missing []string
	switch {
	case ch.Component == "":
		missing = append(missing, "Component")
	case ch.Number == 0:
		missing = append(missing, "Number")
	case ch.SerialNumber == 0:
		missing = append(missing, "SerialNumber")
	case ch.Pop == "":
		missing = append(missing, "Pop")
	case ch.Rack == "":
		missing = append(missing, "Rack")
	case ch.RackUnit == 0:
		missing = append(missing, "RackUnit")
	case ch.CommitDateTime == time.Time{}:
		missing = append(missing, "CommitDateTime")
	case ch.CommitHash == "":
		missing = append(missing, "CommitHash")
	}

	if len(missing) > 0 {
		return errors.Errorf("invalid chassis, missing: %v", strings.Join(missing, ", "))
	}

	return nil
}

// ValidateChildren ensures that we have the correct/expected number of HubNodes and Nodes in the
// Chassis
func (ch *Chassis) ValidateChildren() error {
	switch {
	case len(ch.HubNodes) != 2:
		return errors.Errorf("Unexpected number of HubNodes for Chassis: %v", len(ch.HubNodes))
	case len(ch.Nodes) != 9:
		return errors.Errorf("Unexpected number of Nodes for Chassis: %v", len(ch.Nodes))
	}

	return nil
}
