package spec

import (
	"context"
	"fmt"
	"time"

	pb "code.justin.tv/video-tools/buildsheet-service/api/buildsheet"
	"github.com/pkg/errors"
)

// PBBuildSheetFromChassis converts a Chassis to a proto buildsheet
func PBBuildSheetFromChassis(ctx context.Context, chassis *Chassis) *pb.BuildSheet {
	return &pb.BuildSheet{
		Location: &pb.Location{
			Pop:      chassis.Pop,
			Rack:     chassis.Rack,
			RackUnit: int32(chassis.RackUnit),
		},
		Chassis: &pb.Component{
			Name:   chassis.Component,
			Serial: int64(chassis.SerialNumber),
		},
		HubNodes: pbComponentsFromHubNodes(chassis.HubNodes),
		Nodes:    pbComponentsFromNodes(chassis.Nodes),
	}
}

// ChassisFromPBBuildSheet converts a proto buildsheet to a Chassis
func ChassisFromPBBuildSheet(ctx context.Context, bs *pb.BuildSheet) *Chassis {
	loc := bs.GetLocation()
	return &Chassis{
		Component:    bs.Chassis.GetName(),
		SerialNumber: int(bs.GetChassis().GetSerial()),
		Pop:          loc.GetPop(),
		Rack:         loc.GetRack(),
		RackUnit:     int(loc.GetRackUnit()),
		HubNodes:     hubNodesFromPBComponents(bs.GetHubNodes()),
		Nodes:        nodesFromPBComponents(bs.GetNodes()),
	}
}

// PBChangedBuildSheetsFromChangedBuildSheets converts a set of ChangedBuildSheets to
// their corresponding protobuf representation
func PBChangedBuildSheetsFromChangedBuildSheets(
	ctx context.Context,
	bbs BulkBuildSheetData,
) ([]*pb.ChangedBuildSheet, error) {
	protoCBs := make([]*pb.ChangedBuildSheet, len(bbs))

	for i, cb := range bbs {
		protoCB := &pb.ChangedBuildSheet{
			ChangeType: pbChangeTypeFromChangeType(cb.ChangeType),
			Location: &pb.Location{
				Pop:      cb.Pop,
				Rack:     cb.Rack,
				RackUnit: int32(cb.RackUnit),
			},
		}

		buildSheet, err := NewBuildSheet(cb.Data, cb.Pop, cb.Rack, cb.RackUnit, time.Now())
		if err != nil {
			return nil, errors.Wrap(err, "parsing raw buildsheet from bytes")
		}

		chassis, err := buildSheet.ParseChassis()
		if err != nil {
			return nil, errors.Wrap(err, "parsing chassis info from buildsheet")
		}

		protoCB.BuildSheet = PBBuildSheetFromChassis(ctx, chassis)

		protoCBs[i] = protoCB
	}
	return protoCBs, nil
}

// StrippedChangedBuildSheetsFromPBChangedBuildSheets converts proto ChangedBuildSheets to
// ChangedBuildSheets without the associated data (for publishing to SNS)
func StrippedChangedBuildSheetsFromPBChangedBuildSheets(
	ctx context.Context,
	changes []*pb.ChangedBuildSheet,
) BulkBuildSheetData {
	buildSheets := make(BulkBuildSheetData, len(changes))
	for i, ch := range changes {
		loc := ch.GetLocation()
		buildSheets[i] = &ChangedBuildSheet{
			ChangeType: ch.GetChangeType().String(),
			Pop:        loc.GetPop(),
			Rack:       loc.GetRack(),
			RackUnit:   int(loc.GetRackUnit()),
		}
	}
	return buildSheets
}

func pbComponentsFromHubNodes(hubnodes []*HubNode) []*pb.Component {
	c := make([]*pb.Component, len(hubnodes))
	for i, hubnode := range hubnodes {
		c[i] = &pb.Component{
			Name:         hubnode.Component,
			Serial:       int64(hubnode.SerialNumber),
			Slot:         int32(hubnode.Slot),
			MacAddresses: pbMACsFromMACs(hubnode.MACs),
		}
	}
	return c
}

func hubNodesFromPBComponents(components []*pb.Component) []*HubNode {
	h := make([]*HubNode, len(components))
	for i, component := range components {
		h[i] = &HubNode{
			Component:    component.GetName(),
			SerialNumber: int(component.GetSerial()),
			Slot:         int(component.GetSlot()),
			MACs:         macsFromPBMACs(component.GetMacAddresses()),
		}
	}
	return h
}

func nodesFromPBComponents(components []*pb.Component) []*Node {
	n := make([]*Node, len(components))
	for i, component := range components {
		n[i] = &Node{
			Component:    component.GetName(),
			SerialNumber: int(component.GetSerial()),
			Slot:         int(component.GetSlot()),
			MACs:         macsFromPBMACs(component.GetMacAddresses()),
		}
	}
	return n
}

func pbComponentsFromNodes(nodes []*Node) []*pb.Component {
	c := make([]*pb.Component, len(nodes))
	for i, node := range nodes {
		c[i] = &pb.Component{
			Name:         node.Component,
			Serial:       int64(node.SerialNumber),
			Slot:         int32(node.Slot),
			MacAddresses: pbMACsFromMACs(node.MACs),
		}
	}
	return c
}

func pbMACsFromMACs(macs []*MAC) []*pb.MACAddress {
	addrs := make([]*pb.MACAddress, len(macs))
	for i, m := range macs {
		addrs[i] = &pb.MACAddress{
			Name:    m.Description,
			Address: m.Addr,
		}
	}
	return addrs
}

func macsFromPBMACs(macs []*pb.MACAddress) []*MAC {
	addrs := make([]*MAC, len(macs))
	for i, m := range macs {
		addrs[i] = &MAC{
			Description: m.Name,
			Addr:        m.Address,
		}
	}
	return addrs
}

func pbChangeTypeFromChangeType(changeType string) pb.ChangedBuildSheet_ChangeType {
	switch changeType {
	case "A":
		return pb.ChangedBuildSheet_Added
	case "M":
		return pb.ChangedBuildSheet_Modified
	case "D":
		return pb.ChangedBuildSheet_Deleted
	default:
		panic(fmt.Sprintf("unknown change type %s", changeType))
	}
}
