package spec

import (
	"strconv"
	"strings"

	"github.com/pkg/errors"
)

// HubNode represents one of two switches inside a Kontron Chassis
type HubNode struct {
	ID           int    `db:"id"  json:"-"`
	ChassisID    int    `db:"chassis_id" json:"-"`
	Component    string `db:"component" json:"component"`
	SerialNumber int    `db:"serial_number" json:"serial_number"`
	Slot         int    `db:"slot" json:"slot"`
	MACs         []*MAC `json:"mac_addresses,omitempty"`
	Pop          string `db:"pop" json:"pop,omitempty"`
	Rack         string `db:"rack" json:"rack,omitempty"`
	RackUnit     int    `db:"ru" json:"rack_unit,omitempty"`

	ChassisSerialNumber int `json:"chassis_serial_number,omitempty"`
}

func newHubNode(attrs map[string]string, slot int) (*HubNode, error) {
	hn := HubNode{Slot: slot}

	for attr, val := range attrs {
		switch attr {
		case "Component":
			hn.Component = val
		case "SerialNumber":
			num, err := strconv.Atoi(val)
			if err != nil {
				return nil, errors.Errorf("Failed to parse serial number as integer: %v", val)
			}
			hn.SerialNumber = num
		default:
			mac, err := newMAC(attr, val)
			if err != nil {
				return nil, errors.Wrap(err, "populating hubnode")
			}
			hn.MACs = append(hn.MACs, mac)
		}
	}

	if err := hn.Validate(); err != nil {
		return nil, err
	}

	return &hn, nil
}

// Validate ensures we have no unexpected initialized values for a HubNode object
func (hn *HubNode) Validate() error {
	var missing []string
	switch {
	case hn.Component == "":
		missing = append(missing, "Component")
	case hn.SerialNumber == 0:
		missing = append(missing, "SerialNumber")
	case hn.Slot == 0:
		missing = append(missing, "Slot")
	}

	if len(missing) > 0 {
		return errors.Errorf("invalid HubNode, missing: %v", strings.Join(missing, ", "))
	}

	if hn.Slot > 2 {
		return errors.Errorf("invalid slot number for HubNode: %v", hn.Slot)
	}

	return nil
}
