package spec

import (
	"net"
	"strings"

	"github.com/pkg/errors"
)

// MAC represents a MAC address and its descriptor, for a given HubNode or Node in a Kontron
// Chassis
type MAC struct {
	ParentID    int    `db:"parent_id" json:"-"`
	MACType     string `db:"type" json:"-"`
	Description string `db:"description" json:"description"`
	Addr        string `db:"mac" json:"mac"`
}

func newMAC(description, addr string) (*MAC, error) {
	mac := &MAC{
		Description: description,
		Addr:        addr,
	}

	if err := mac.Validate(); err != nil {
		return nil, err
	}

	return mac, nil
}

// Validate ensures we have no unexpected initialized values for a MAC object
func (m *MAC) Validate() error {
	if _, err := net.ParseMAC(m.Addr); err != nil {
		return errors.Wrap(err, "parsing address for MAC")
	}

	var missing []string
	switch {
	case m.Description == "":
		missing = append(missing, "Description")
	// we should never hit this, but adding it for completeness
	case m.Addr == "":
		missing = append(missing, "Addr")
	}

	if len(missing) > 0 {
		return errors.Errorf("invalid MAC, missing: %v", strings.Join(missing, ", "))
	}

	return nil
}
