package spec

import (
	"strconv"
	"strings"

	"github.com/pkg/errors"
)

// Node represents one of two nodes on a sled inside a Kontron Chassis
type Node struct {
	ID           int    `db:"id"  json:"-"`
	ChassisID    int    `db:"chassis_id" json:"-"`
	Component    string `db:"component" json:"component"`
	SerialNumber int    `db:"serial_number" json:"serial_number"`
	Slot         int    `db:"slot" json:"slot"`
	MACs         []*MAC `json:"mac_addresses,omitempty"`
	Pop          string `db:"pop" json:"pop,omitempty"`
	Rack         string `db:"rack" json:"rack,omitempty"`
	RackUnit     int    `db:"ru" json:"rack_unit,omitempty"`

	ChassisSerialNumber  int `json:"chassis_serial_number,omitempty"`
	Hubnode1SerialNumber int `json:"hubnode_1_serial_number,omitempty"`
	Hubnode2SerialNumber int `json:"hubnode_2_serial_number,omitempty"`
}

func newNode(attrs map[string]string, slot int) (*Node, error) {
	n := Node{Slot: slot}
	for attr, val := range attrs {
		switch attr {
		case "Component":
			n.Component = val
		case "SerialNumber":
			num, err := strconv.Atoi(val)
			if err != nil {
				return nil, errors.Errorf("Failed to parse serial number as integer: %v", val)
			}
			n.SerialNumber = num
		default:
			mac, err := newMAC(attr, val)
			if err != nil {
				return nil, errors.Wrap(err, "populating Node")
			}
			n.MACs = append(n.MACs, mac)
		}
	}

	if err := n.Validate(); err != nil {
		return nil, err
	}

	return &n, nil
}

// Validate ensures we have no unexpected initialized values for a Node object
func (n *Node) Validate() error {
	var missing []string
	switch {
	case n.Component == "":
		missing = append(missing, "Component")
	case n.SerialNumber == 0:
		missing = append(missing, "SerialNumber")
	case n.Slot == 0:
		missing = append(missing, "Slot")
	}

	if len(missing) > 0 {
		return errors.Errorf("invalid Node, missing: %v", strings.Join(missing, ", "))
	}

	if n.Slot > 9 {
		return errors.Errorf("invalid slot for node: %v", n.Slot)
	}

	return nil
}
