// tslint:disable-next-line: no-submodule-imports
import 'source-map-support/register';

import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

import { EcrRepositoryStack } from './stacks/ecr-repository';
import { ECSClusterStack } from './stacks/ecs-cluster';
import { ECSFargateServiceStack } from './stacks/ecs-fargate-service';
import { VpcStack } from './stacks/vpc';

interface EnvProps {
  prefix: string;
  env: cdk.Environment;
  ec2AsgInstanceType: ec2.InstanceType;
  ec2AsgCapacity: number;
  vpcId: string;
  vpcPrivSubnets: ec2.SubnetAttributes[];
}

const app = new cdk.App();

createStacks(app, {
  prefix: 'Dev',
  env: {
    account: '733778597747', // beta account 733778597747
    region: 'us-west-2',
  },
  ec2AsgInstanceType: new ec2.InstanceType('m5.large'),
  ec2AsgCapacity: 0, // used for dev02 test instance
  vpcId: 'vpc-0a715ed99bf9085f4', // cmon allowlist service vpc
  vpcPrivSubnets: [
    { subnetId: 'subnet-0304befc55bc7f1e6', availabilityZone: 'us-west-2a', routeTableId: 'rtb-0284d756276826ee0' }, // 10.0.96.0/19 // allowlist private vpc us-west-2a
    { subnetId: 'subnet-043fd3286d96c09e0', availabilityZone: 'us-west-2b', routeTableId: 'rtb-009085b3a29a83592' }, // 10.0.128.0/19 // allowlist private vpc us-west-2b
    { subnetId: 'subnet-0d4c6b03efd3ca622', availabilityZone: 'us-west-2c', routeTableId: 'rtb-043560540a88ee9f8' }, // 10.0.160.0/19 // allowlist private vpc us-west-2c
  ]
});

// createStacks for each environment
function createStacks(scope: cdk.App, props: EnvProps) {
  const { prefix, env } = props;

  const vpcStack = new VpcStack(scope, prefix + 'Vpc', {
    env,
    vpcId: props.vpcId,
    vpcPrivSubnets: props.vpcPrivSubnets,
  });

  const ecrRepoStack = new EcrRepositoryStack(app, prefix + 'LoadtesterRepo', { env, repositoryName: 'loadtester' }); // repo name as used by scripts/docker_push.sh

  const clusterStack = new ECSClusterStack(app, prefix + 'LoadtesterECSCluster', {
    env,
    vpc: vpcStack.vpc,
    ec2Asg: {
      instanceType: props.ec2AsgInstanceType,
      minCapacity: props.ec2AsgCapacity,
      maxCapacity: props.ec2AsgCapacity,
    },
  });

  new ECSFargateServiceStack(app, prefix + 'LoadtesterService', {
    env,
    vpc: vpcStack.vpc,
    cluster: clusterStack.cluster,
    ecrRepo: ecrRepoStack.ecrRepo
  });
}