import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecr from '@aws-cdk/aws-ecr';
import * as ecs from '@aws-cdk/aws-ecs';
import * as iam from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';

interface ECSServiceStackProps extends cdk.StackProps {
    vpc: ec2.IVpc;
    cluster: ecs.ICluster;
    ecrRepo: ecr.IRepository;
}

export class ECSFargateServiceStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ECSServiceStackProps) {
    super(scope, id, props);

    const taskDefinition = new ecs.FargateTaskDefinition(this, 'TaskDefinition', {
      cpu: 4096, // 4 vCPU
      memoryLimitMiB: 8192, // 8 GB (minimum available for the cpu value)
    });

    // policy used by S2S client: https://wiki.twitch.com/pages/viewpage.action?pageId=183990900
    taskDefinition.addToTaskRolePolicy(
        new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: ['arn:aws:iam::180116294062:role/malachai/*'],
        })
    );

    taskDefinition.addToTaskRolePolicy(
      // Send custom metrics to CloudWatch for TwitchTelemetry
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        resources: ['*'],
      })
    );

    const container = taskDefinition.addContainer('TaskContainer', {
        image: ecs.ContainerImage.fromEcrRepository(props.ecrRepo, 'latest'), // Requires props.ecrRepo to exist and have a registered docker image
        logging: new ecs.AwsLogDriver({ streamPrefix: 'loadtester' }),
        memoryReservationMiB: 1024, // 1 GB
        environment: {
          TARGETENV: "dev",
        }
    });

    container.addUlimits({
      name: ecs.UlimitName.NOFILE, // number of open files, needed to handle socket connections
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Service
    new ecs.FargateService(this, 'TaskService', {
      serviceName: `loadtester`,
      cluster: props.cluster,
      taskDefinition,
      assignPublicIp: false,
      desiredCount: 8, // updated through /scripts/run.sh
    });
  }
}

