import * as ec2 from '@aws-cdk/aws-ec2';
import * as cdk from '@aws-cdk/core';

interface VpcStackProps extends cdk.StackProps {
  vpcId: string; // id to import the vpc
  vpcPrivSubnets: ec2.SubnetAttributes[]; // to import the 3 private subnets (one for each availablility zone)
}

export class VpcStack extends cdk.Stack {
  public readonly vpc: ec2.IVpc;
  public readonly vpcPrivSubnets: ec2.ISubnet[];

  constructor(scope: cdk.Construct, name: string, props: VpcStackProps) {
    super(scope, name, props);

    // Import already existing VPC
    this.vpc = ec2.Vpc.fromLookup(this, 'Vpc', { vpcId: props.vpcId });

    // In theory, Vpc.fromLookup should also import the subnets,
    // but for some reason this.vpc.privateSubnets is empty.
    // Import the subnets manually and expose them on a stack property.
    this.vpcPrivSubnets = props.vpcPrivSubnets.map((subnetAttrs, i) =>
      ec2.Subnet.fromSubnetAttributes(this, `PrivSubnet${i}`, subnetAttrs)
    );
  }
}