﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Threading.Tasks;

namespace Twitch.Authorization
{
    public class ClientAuth : IDisposable
    {
        public string AccessToken { get; private set; } = "";

        private IntPtr clientAuth;

        public Task<string> PerformAsync(string clientId, string redirectUri, IEnumerable<string> scopes)
        {
            Utility.CheckArgument(nameof(clientId), clientId);
            Utility.CheckArgument(nameof(redirectUri), redirectUri);
            if (clientAuth != IntPtr.Zero)
            {
                throw new InvalidOperationException("client auth already started");
            }
            var s = Utility.ComposeScopes(scopes);
            clientAuth = NativeMethods.CreateClientAuth();
            AccessToken = null;
            return InternalPerformAsync(clientId, redirectUri, s);
        }

        private async Task<string> InternalPerformAsync(string clientId, string redirectUri, string scopes)
        {
            try
            {
                AccessToken = await Task.Run(() =>
                {
                    return NativeMethods.DoClientAuth(clientAuth, clientId, redirectUri, scopes);
                });
            }
            finally
            {
                NativeMethods.DeleteClientAuth(ref clientAuth);
            }
            return AccessToken;
        }

        public void Cancel()
        {
            if (clientAuth != IntPtr.Zero)
            {
                NativeMethods.CancelClientAuth(clientAuth);
            }
        }

        #region IDisposable Support

        private bool isDisposed = false;

        protected virtual void Dispose(bool disposing)
        {
            if (!isDisposed)
            {
                if (disposing)
                {
                    // Dispose of managed state.
                }

                // Free unmanaged resources.
                NativeMethods.DeleteClientAuth(ref clientAuth);

                isDisposed = true;
            }
        }

        ~ClientAuth()
        {
            Dispose(false);
        }

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        #endregion

        private static class NativeMethods
        {
            [DllImport("Twitch.Authorization.ClientAuth", CallingConvention = CallingConvention.Cdecl)]
            internal static extern IntPtr CreateClientAuth();

            [DllImport("Twitch.Authorization.ClientAuth", CallingConvention = CallingConvention.Cdecl, CharSet = CharSet.Unicode)]
            internal static extern string DoClientAuth(IntPtr clientAuth, string clientId, string redirectUri, string scopes);

            [DllImport("Twitch.Authorization.ClientAuth", CallingConvention = CallingConvention.Cdecl)]
            internal static extern void CancelClientAuth(IntPtr clientAuth);

            [DllImport("Twitch.Authorization.ClientAuth", CallingConvention = CallingConvention.Cdecl)]
            internal static extern void DeleteClientAuth(ref IntPtr clientAuth);
        }
    }
}
