﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;
using Newtonsoft.Json.Linq;

namespace Twitch.Authorization
{
    public class DeviceAuth
    {
        public string UserCode { get; private set; }
        public string VerificationUri { get; private set; }

        private const string deviceUrl = "https://id.twitch.tv/oauth2/device";
        private const string tokenUrl = "https://id.twitch.tv/oauth2/token";
        private static readonly HttpClient client = new HttpClient();
        private readonly string clientId;
        private readonly string deviceCode;
        private readonly TimeSpan delay;

        private DeviceAuth(string clientId, string deviceCode, int interval, string userCode, string verificationUri)
        {
            this.clientId = clientId;
            this.deviceCode = deviceCode;
            delay = TimeSpan.FromSeconds(interval);
            UserCode = userCode;
            VerificationUri = verificationUri;
        }

        public static async Task<DeviceAuth> StartAuthAsync(string clientId, IEnumerable<string> scopes, CancellationToken token = default(CancellationToken))
        {
            // Request codes for Device Grant flow.

            // Send the request.
            var request = new HttpRequestMessage(HttpMethod.Post, deviceUrl);
            var s = WebUtility.UrlEncode(Utility.ComposeScopes(scopes));
            var content = $"client_id={WebUtility.UrlEncode(clientId)}&scopes={s}";
            request.Content = new StringContent(content);
            request.Headers.Add("Client-ID", clientId);
            var response = await client.SendAsync(request, token);
            response.EnsureSuccessStatusCode();

            // Parse the response as JSON.
            var responseBody = await response.Content.ReadAsStringAsync();
            var jsonVal = JObject.Parse(responseBody);
            var deviceCode = jsonVal["device_code"].Value<string>();
            var interval = jsonVal["interval"].Value<int>();
            var userCode = jsonVal["user_code"].Value<string>();
            var verificationUri = jsonVal["verification_uri"].Value<string>();
            if (!deviceCode.Any() || !userCode.Any() || !verificationUri.Any())
            {
                throw new Exception("invalid response from server");
            }

            return new DeviceAuth(clientId, deviceCode, interval, userCode, verificationUri);
        }

        public async Task<KeyValuePair<string, string>> FinishAuthAsync(CancellationToken token = default(CancellationToken))
        {
            // Start a task to poll the token endpoint to complete the Device Grant flow.
            KeyValuePair<string, string> rv;
            for (; ; )
            {
                await Task.Delay(delay, token);

                // Request token for Device Grant flow.

                // Send the request.
                var request = new HttpRequestMessage(HttpMethod.Post, tokenUrl);
                var client_id = WebUtility.UrlEncode(clientId);
                var device_code = WebUtility.UrlEncode(deviceCode);
                var content = $"client_id={client_id}&device_code={device_code}&grant_type=urn%3Aietf%3Aparams%3Aoauth%3Agrant-type%3Adevice_code";
                request.Content = new StringContent(content);
                request.Headers.Add("Client-ID", clientId);
                var response = await client.SendAsync(request, token);
                response.EnsureSuccessStatusCode();

                // Parse the response as JSON.
                var responseBody = await response.Content.ReadAsStringAsync();
                var jsonVal = JObject.Parse(responseBody);
                var error = jsonVal["message"].Value<string>();
                if (error == "authorization_pending")
                {
                    continue;
                }
                if (error == "invalid device code")
                {
                    break;
                }
                if (error.Any())
                {
                    throw new Exception(error);
                }
                var accessToken = jsonVal["access_token"].Value<string>();
                var refreshToken = jsonVal["refresh_token"].Value<string>();
                if (!accessToken.Any() || !refreshToken.Any())
                {
                    throw new Exception(responseBody);
                }
                rv = new KeyValuePair<string, string>(accessToken, refreshToken);
            }
            return rv;
        }
    }
}
