#include "pch.h"
#include "../../base-sdk/Shared/Json.h"

using namespace std::chrono_literals;
using namespace Twitch;

namespace {
	tstring const deviceUrl = _T("https://id.twitch.tv/oauth2/device");
	tstring const tokenUrl = _T("https://id.twitch.tv/oauth2/token");

	std::vector<char> MakePostData(std::function<void(std::ostream&)> fn) {
		std::stringstream ss;
		fn(ss);
		auto s = ss.str();
		return std::vector<char>(s.cbegin(), s.cend());
	}
}

std::future<DeviceAuth::Response> DeviceAuth::StartAuth(tstring const& clientId, tstring const& scopes) {
	if(promise) {
		throw TwitchException(FromPlatformError(ERROR_IO_PENDING));
	}
	promise = std::make_unique<std::remove_reference_t<decltype(*promise)>>();

	auto fn = [this, clientId = tstring(clientId), scopes = tstring(scopes)] {
		// Request codes for Device Grant flow.
		auto request = HttpRequest(clientId);
		auto postData = MakePostData([clientId, scopes](std::ostream& os) {
			os << "client_id=" << FromTstring(UrlEncode(clientId)) << "&scopes=" << FromTstring(UrlEncode(scopes));
		});
		auto response = request.Post(deviceUrl, postData, true);
		ThrowIfResponseFailed(response);
		auto body = response.Response;
		body.push_back('\0');
		std::string parseError;
		auto jsonVal = Json::Parse(std::string_view(body.data(), body.size()), parseError);
		if(!parseError.empty()) {
			DebugWriteLine(_T("[DeviceAuth::StartAuth] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}
		auto deviceCode = jsonVal["device_code"].string_value();
		//auto expiresIn = jsonVal["expires_in"].int_value();
		auto interval = jsonVal["interval"].int_value();
		auto userCode = jsonVal["user_code"].string_value();
		auto verificationUri = jsonVal["verification_uri"].string_value();
		if(deviceCode.empty() || userCode.empty() || verificationUri.empty()) {
			DebugWriteLine(_T("[DeviceAuth::StartAuth] unexpected JSON response:  %s"), ToTstring(body.data()).c_str());
			throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
		}

		// Start a task to poll the token endpoint to complete the Device Grant flow.
		postData = MakePostData([clientId, deviceCode](std::ostream& os) {
			os << "client_id=" << FromTstring(UrlEncode(clientId)) << "&device_code=" <<
				FromTstring(UrlEncode(ToTstring(deviceCode))) <<
				"&grant_type=urn%3Aietf%3Aparams%3Aoauth%3Agrant-type%3Adevice_code";
		});
		auto fn_ = [this, clientId, duration = interval * 1s, postData = std::move(postData)] {
			auto future = promise->get_future();
			std::pair<tstring, tstring> rv;
			while(future.wait_for(duration) == std::future_status::timeout) {
				// Request token for Device Grant flow.
				auto request = HttpRequest(clientId);
				auto response = request.Post(tokenUrl, postData);
				if(response.Result) {
					throw TwitchException(response.Result);
				}
				auto body = response.Response;
				body.push_back('\0');
				std::string parseError;
				auto jsonVal = Json::Parse(std::string_view(body.data(), body.size()), parseError);
				if(!parseError.empty()) {
					DebugWriteLine(_T("[DeviceAuth::StartAuth.poll] JSON parsing failed:  %s"), ToTstring(parseError).c_str());
					throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
				}

				// Check the results.
				auto error = jsonVal["message"].string_value();
				if(error == "authorization_pending") {
					continue;
				}
				if(error == "invalid device code") {
					break;
				}
				if(!error.empty()) {
					DebugWriteLine(_T("[DeviceAuth::StartAuth.poll] request failed:  %s"), ToTstring(error).c_str());
					throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
				}
				auto accessToken = jsonVal["access_token"].string_value();
				auto refreshToken = jsonVal["refresh_token"].string_value();
				if(accessToken.empty() || refreshToken.empty()) {
					DebugWriteLine(_T("[DeviceAuth::StartAuth] unexpected JSON response:  %s"), ToTstring(body.data()).c_str());
					throw TwitchException(FromPlatformError(ERROR_INVALID_RESPONSE));
				}
				rv.first = ToTstring(accessToken);
				rv.second = ToTstring(refreshToken);
				break;
			}
			return rv;
		};
		return Response{ ToTstring(userCode), ToTstring(verificationUri), std::async(std::launch::async, fn_) };
	};
	return std::async(std::launch::deferred, fn);
}

void DeviceAuth::Cancel() {
	if(promise) {
		promise->set_value();
	}
}
