// Copyright (c) 2019-2021, Twitch Interactive, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.

#pragma once

#include <cassert>
#include <ctime>
#include <condition_variable>
#include <exception>
#include <functional>
#include <future>
#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <utility>
#include <vector>

namespace Twitch {
#if defined(_WIN32) && !defined(_XBOX_ONE)
	class ApplicationAuth;
#endif
#ifndef __NX__
	class BrowserAuth;
	class ServerAuth;
# ifndef _XBOX_ONE
	class ClientAuth;
# endif
#endif
	class DeviceAuth;
}

#if defined(_WIN32) && !defined(_XBOX_ONE)
class Twitch::ApplicationAuth {
public:
	std::future<tstring> StartAuth(tstring const& clientId, tstring const& clientSecret, tstring const& scopes);
};
#endif

#ifndef __NX__
class Twitch::BrowserAuth {
public:
	BrowserAuth();
	~BrowserAuth();

	template<typename T>
	static tstring ConstructScopes(T const& scopes) {
		tstring scope;
		for(auto s : scopes) {
			scope.append(s);
			scope.append(1, tstring::value_type(' '));
		}
		scope.pop_back();
		return scope;
	}
	std::future<tstring> Launch(tstring const& clientId, tstring const& redirectUri, tstring const& responseType, tstring const& scopes, tstring& state, string_t scheme = nullptr);
	void Stop(tstring const& accessToken);
	tstring GetToken() const;

private:
	std::mutex tokenAccessor;
	using TokenLock = std::unique_lock<decltype(tokenAccessor)>;
# ifndef __ORBIS__
	std::condition_variable cv;
	bool isReady;
# endif
	tstring token;

	std::future<tstring> Launch(tstring const& url, tstring const& state, string_t scheme);
};

class Twitch::ServerAuth : Twitch::BrowserAuth {
public:
	ServerAuth();
	~ServerAuth();

	std::future<tstring> StartAuth(tstring const& clientId, tstring const& redirectUri, tstring const& scopes, tstring& state);
	void Finish(tstring const& accessToken, tstring const& refreshToken);
	void Cancel();
	tstring GetAccessToken() const { return GetToken(); }
	tstring GetRefreshToken() const;

	__declspec(property(get = GetAccessToken)) tstring const AccessToken;
	__declspec(property(get = GetRefreshToken)) tstring const RefreshToken;

private:
	tstring refreshToken;
};

# ifndef _XBOX_ONE
class Twitch::ClientAuth : Twitch::BrowserAuth {
public:
	ClientAuth();
	~ClientAuth();

	std::future<tstring> StartAuth(tstring const& clientId, tstring const& redirectUri, tstring const& scopes);
	void Cancel();
	tstring GetAccessToken() const { return GetToken(); }

	__declspec(property(get = GetAccessToken)) tstring const AccessToken;

#  ifdef _WIN32
private:
	std::shared_ptr<struct DdeServer> ddeServer;
#  endif
};
# endif
#endif

class Twitch::DeviceAuth {
public:
	struct Response {
		tstring userCode;
		tstring verificationUrl;
		std::future<std::pair<tstring, tstring>> tokenTask;
	};

	std::future<Response> StartAuth(tstring const& clientId, tstring const& scopes);
	void Cancel();

private:
	std::unique_ptr<std::promise<void>> promise;
	bool isCanceled;
};
