#pragma once

class HttpsConnection {
#ifdef _DEBUG
	static constexpr size_t bufferSize = 64;
#else
	static constexpr size_t bufferSize = 2048;
#endif

public:
	using vector = std::vector<char>;

	static vector const EmptyData;

	HttpsConnection(std::string const& hostName, unsigned short port = 443) : hostName(hostName), tcpSocket(-1), port(port) {}
	~HttpsConnection() {
		Close();
	}
	bool Open() {
		if(CreateSocket() && ConfigureSsl() && ConfigureSslConnection() && ConfigureHostName() && PerformSslHandshake()) {
			return true;
		} else {
			Close();
			return false;
		}
	}
	std::tuple<int, vector> Delete(std::string const& path, std::string const& headers) {
		return Send("DELETE", path, headers, EmptyData);
	}
	std::tuple<int, vector> Get(std::string const& path, std::string const& headers) {
		return Send("GET", path, headers, EmptyData);
	}
	std::tuple<int, vector> Post(std::string const& path, std::string const& headers, vector const& data) {
		return Send("POST", path, headers, data);
	}
	std::tuple<int, vector> Put(std::string const& path, std::string const& headers, vector const& data) {
		return Send("PUT", path, headers, data);
	}
	std::tuple<int, vector> Send(std::string const& verb, std::string const& path, std::string const& headers, vector const& data) {
		if(tcpSocket == -1) {
			if(!Open()) {
				return std::make_tuple(-1, vector());
			}
		}

		std::stringstream ss;
		ss << verb << ' ' << path << " HTTP/1.0\r\nHost: " << hostName << "\r\n" << headers << "\r\n";
		ss.write(data.data(), data.size());
		auto s = ss.str();
		int sentBytes = 0;
		nn::Result result = m_SslConnection.Write(s.c_str(), &sentBytes, static_cast<std::uint32_t>(s.size()));
		if(result.IsFailure() || sentBytes < 0) {
			DebugWriteLine("[HttpsConnection::Send] nn::ssl::Connection.Write failed:  %d", result.GetDescription());
		}

		vector response;
		for(;;) {
			char buffer[bufferSize];
			int receivedBytes = 0;
			result = m_SslConnection.Read(buffer, &receivedBytes, sizeof(buffer));
			if(result.IsFailure() || receivedBytes < 0) {
				DebugWriteLine("[HttpsConnection::Send] nn::ssl::Connection.Read failed:  %d", result.GetDescription());
				break;
			}
			if(receivedBytes == 0) {
				// The server closed the connection.
				Close();
				break;
			}
			response.insert(response.end(), buffer, buffer + receivedBytes);
		}

		return std::make_tuple(result.IsSuccess() ? 0 : -1, response);
	}
	void Close() {
		// Close the socket.
		if(tcpSocket >= 0) {
			nn::socket::Close(tcpSocket);
			tcpSocket = -1;
		}

		// Close the SSL connection.
		nn::ssl::SslConnectionId connectionId;
		if(m_SslConnection.GetConnectionId(&connectionId).IsSuccess() && connectionId != 0) {
			m_SslConnection.Destroy();
			tcpSocket = -1;
		}

		// Destroy the SSL context.
		nn::ssl::SslContextId contextId;
		if(m_SslContext.GetContextId(&contextId).IsSuccess() && contextId != 0) {
			m_SslContext.Destroy();
		}
	}

private:
	static constexpr std::uint32_t MaxAsyncDoHandshakeRetry = 30;
	nn::ssl::Context m_SslContext;
	nn::ssl::Connection m_SslConnection;
	std::string hostName;
	int tcpSocket;
	unsigned short port;

	bool CreateSocket() {
		if(tcpSocket < 0) {
			nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(hostName.c_str());
			if(pHostEnt == nullptr) {
				DebugWriteLine("[HttpsConnection::CreateSocket] nn::socket::GetHostEntByName failed:  %d", errno);
				return false;
			}

			tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
			if(tcpSocket < 0) {
				DebugWriteLine("[HttpsConnection::CreateSocket] nn::socket::Socket failed:  %d", errno);
				return false;
			}

			nn::socket::SockAddrIn serverAddr{};
			serverAddr.sin_addr.S_addr = reinterpret_cast<nn::socket::InAddr*>(pHostEnt->h_addr_list[0])->S_addr;
			serverAddr.sin_family = nn::socket::Family::Af_Inet;
			serverAddr.sin_port = nn::socket::InetHtons(port);
			int result = nn::socket::Connect(tcpSocket, reinterpret_cast<nn::socket::SockAddr*>(&serverAddr), sizeof(serverAddr));
			if(result < 0) {
				DebugWriteLine("[HttpsConnection::CreateSocket] nn::socket::Connect failed:  %d", errno);
				return false;
			}
		}
		return true;
	}

	bool ConfigureSsl() {
		nn::Result result = m_SslContext.Create(nn::ssl::Context::SslVersion::SslVersion_Auto);
		if(result.IsFailure()) {
			DebugWriteLine("[HttpsConnection::ConfigureSsl] nn::ssl::Context.Create failed:  %d", result.GetDescription());
			return false;
		}
		nn::ssl::SslContextId contextId;
		result = m_SslContext.GetContextId(&contextId);
		if(result.IsFailure()) {
			m_SslContext.Destroy();
			DebugWriteLine("[HttpsConnection::ConfigureSsl] nn::ssl::Context.GetContextId failed:  %d", result.GetDescription());
			return false;
		}
		return true;
	}

	bool ConfigureSslConnection() {
		nn::Result result = m_SslConnection.Create(&m_SslContext);
		if(result.IsFailure()) {
			DebugWriteLine("[HttpsConnection::ConfigureSslConnection] nn::ssl::Connection.Create failed:  %d", result.GetDescription());
			return false;
		}
		result = m_SslConnection.SetSocketDescriptor(tcpSocket);
		if(result.IsFailure()) {
			DebugWriteLine("[HttpsConnection::ConfigureSslConnection] nn::ssl::Connection.SetSocketDescriptor failed:  %d", result.GetDescription());
			return false;
		}
		tcpSocket = -2;
		return true;
	}

	bool PerformSslHandshake() {
		nn::Result result = nn::ResultSuccess();
		bool isSuccess = false;
		std::uint32_t asyncHandshakeCount = 0;
		do {
			result = m_SslConnection.DoHandshake();
			if(result.IsSuccess()) {
				isSuccess = true;
				break;
			}
			if(nn::ssl::ResultBufferTooShort::Includes(result)) {
				// Failed to store server certificate, but it is ok to continue
			} else if(nn::ssl::ResultIoWouldBlock::Includes(result)) {
				if(asyncHandshakeCount++ < MaxAsyncDoHandshakeRetry) {
					nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
					continue;
				} else {
					DebugWriteLine("[HttpsConnection::PerformSslHandshake] nn::ssl::Connection.DoHandshake timed out");
					break;
				}
			} else {
				DebugWriteLine("[HttpsConnection::PerformSslHandshake] nn::ssl::Connection.DoHandshake failed:  %d",
					result.GetDescription());
				break;
			}
		} while(NN_STATIC_CONDITION(true));
		return isSuccess;
	}

	bool ConfigureHostName() {
		bool isSuccess = true;
		auto hostNameLen = static_cast<std::uint32_t>(hostName.size());
		nn::Result result = m_SslConnection.SetHostName(hostName.c_str(), hostNameLen);
		if(result.IsFailure()) {
			DebugWriteLine("[HttpsConnection::ConfigureHostName] nn::ssl::Connection.SetHostName failed:  %d",
				result.GetDescription());
			return false;
		}
		return isSuccess;
	}
};

vector const HttpsConnection::EmptyData;
