#include "pch.h"
#include <nn/ssl.h>
#include <nn/socket.h>
#include <curl/curl.h>
#include <Internal.h>
#include <HttpUtil.h>

using namespace Twitch;

using vector = std::vector<char>;

namespace {
#include "HttpsConnection.inl"

	struct Internets {
		Internets(vector&& data) : data(data), statusCode(0) {}
		Internets() = delete;
		Internets(Internets const&) = delete;
		Internets(Internets&&) = default;
		~Internets() = default;
		static vector GetData(uintptr_t handle) {
			auto* p = reinterpret_cast<Internets*>(handle);
			p->SetHeaderIndices();
			if (p->headerEnd == p->data.cend()) {
				return vector();
			}
			return vector(p->headerEnd + 2, p->data.cend());
		}
		static std::string GetHeaders(uintptr_t handle) {
			auto* p = reinterpret_cast<Internets*>(handle);
			p->SetHeaderIndices();
			if (p->headerEnd == p->data.cend()) {
				return std::string();
			}
			return std::string(p->headerBegin, p->headerEnd);
		}
		static int GetStatusCode(uintptr_t handle) {
			auto* p = reinterpret_cast<Internets*>(handle);
			if (!p->statusCode) {
				auto const it = std::find(p->data.cbegin(), p->data.cend(), ' ');
				auto statusCode = it != p->data.cend() ? strtol(&*it + 1, nullptr, 10) : 0;
				if (!statusCode) {
					statusCode = -1;
				}
				p->statusCode = statusCode;
			}
			return p->statusCode;
		}

	private:
		static std::string const headerSeparator;
		vector data;
		vector::const_iterator headerBegin, headerEnd;
		int statusCode;

		void SetHeaderIndices() {
			if (headerEnd == vector::const_iterator()) {
				headerEnd = data.cend();
				auto it = std::find(data.cbegin(), data.cend(), '\n');
				if (it != data.cend()) {
					++it;
					headerBegin = it;
					it = std::search(it, data.cend(), headerSeparator.cbegin(), headerSeparator.cend());
					if (it != data.cend()) {
						headerEnd = it + 2;
					}
				}
			}
		}
	};

	std::unordered_map<tstring, HttpsConnection> connections;
}

std::string const Internets::headerSeparator = "\r\n\r\n";

std::tuple<int, uintptr_t> Platform::StartHttpRequest(string_t url, HttpRequestType requestType,
	std::chrono::milliseconds timeout, std::vector<HttpParam> const& requestHeaders, vector const& requestBody) {
	UNREFERENCED_PARAMETER(timeout); // TODO:  use timeout.
	// Validate the protocol.
	auto const uri = Twitch::Uri(url);
	auto const protocol = uri.GetProtocol();
	if (protocol.size() < 4 || !std::equal(protocol.cbegin(), protocol.cbegin() + 4, _T("http"))) {
		return std::make_tuple(-1, 0);
	} else if (protocol.size() > 5 || protocol[4] != _T('s')) {
		return std::make_tuple(-1, 0);
	}

	// Extract the necessary URL parts.
	auto const hostName = uri.GetHostName();
	std::uint16_t port;
	if (!uri.GetPort(port)) {
		port = protocol == _T("https") ? 443 : 80;
	}
	auto const authority = hostName + _T(':') + to_tstring(port);
	auto const path = uri.GetPath();

	// Determine the HTTP verb.
	std::string verb;
	switch (requestType) {
	case HttpRequestType::HTTP_PUT_REQUEST:
		verb = "PUT";
		break;
	case HttpRequestType::HTTP_POST_REQUEST:
		verb = "POST";
		break;
	case HttpRequestType::HTTP_DELETE_REQUEST:
		verb = "DELETE";
		break;
	default:
		verb = "GET";
		break;
	}

	// Get or create a HTTP connection.
	auto it = std::find_if(connections.begin(), connections.end(), [&authority](auto pair) {
		return pair.first == authority;
	});
	if (it == connections.cend()) {
		it = connections.emplace(authority, HttpsConnection(hostName, port)).first;
	}

	// Build the headers.
	auto const requestHeaderData = BuildHttpHeader(requestHeaders);

	// Perform the operation.
	if (!it->second.Open()) {
		return std::make_tuple(-1, 0);
	}
	int result;
	vector data;
	std::tie(result, data) = it->second.Send(verb, path, requestHeaderData, requestBody);

	return std::make_tuple(result, reinterpret_cast<uintptr_t>(new Internets(std::move(data))));
}

int Platform::GetHttpResponseResultCode(uintptr_t handle) {
	// Extract the status code from the response.
	return Internets::GetStatusCode(handle);
}

std::tuple<int, std::map<tstring, tstring>> Platform::GetHttpResponseHeaders(uintptr_t handle) {
	// Extract the headers.
	auto const headers = Internets::GetHeaders(handle);
	if (headers.empty()) {
		return std::make_tuple(-1, std::map<tstring, tstring>());
	}

	// Parse the headers into a dictionary.
	std::map<tstring, tstring> responseHeaders;
	ParseHeaders(headers.c_str(), responseHeaders);
	return std::make_tuple(0, responseHeaders);
}

std::tuple<int, vector> Platform::GetHttpResponse(uintptr_t handle, std::chrono::milliseconds /*timeout*/) {
	// Read the HTTP response and invoke the call-back.
	auto const data = Internets::GetData(handle);
	return std::make_tuple(0, data);
}

void Platform::FinishHttpRequest(uintptr_t& handle, std::chrono::milliseconds /*timeout*/) {
	auto handle_ = __sync_swap(&handle, std::remove_reference_t<decltype(handle)>());
	std::unique_ptr<Internets>(reinterpret_cast<Internets*>(handle_));
}
