#include "pch.h"
#include <nn/websocket.h>

#pragma comment(lib, "nn_websocket")

using namespace Twitch;

namespace {}

struct WebSocket::WebSocketImpl {
	ClosedFn OnClosed = DefaultClosedFn;
	ReceivedFn OnReceived = DefaultReceivedFn;
	bool GetIsOpen() const { return isOpen; }
	__declspec(property(get = GetIsOpen)) bool const IsOpen;

	WebSocketImpl() = default;
	WebSocketImpl(WebSocketImpl const&) = delete;
	WebSocketImpl(WebSocketImpl&&) = default;
	WebSocketImpl& operator=(WebSocketImpl const&) = delete;
	WebSocketImpl& operator=(WebSocketImpl&&) = default;
	~WebSocketImpl() {
		OnClosed = DefaultClosedFn;
		OnReceived = DefaultReceivedFn;
		Close();
	}

	void Close() {
		if (__sync_swap(&isOpen, false)) {
			pair->second.Close();
			while (isOpen) {
				pair->second.Perform();
			}
			pair.reset();
		}
	}

	int Open(string_t url, std::chrono::milliseconds /*timeout*/) {
		if (IsOpen) {
			DebugWriteLine(_T("[WebSocketImpl::Open] warning:  Open called without preceding Close"));
			assert(!IsOpen);
			Close();
		}

		// Make the connection and socket.
		pair = std::make_unique<std::remove_reference_t<decltype(*pair)>>();
		pair->second.Initialize();
		pair->second.SetOpenEventHandler(OpenEventHandler, this);
		pair->second.SetMessageEventHandler(MessageEventHandler, this);
		pair->second.SetCloseEventHandler(CloseEventHandler, this);
		pair->second.SetErrorEventHandler(ErrorEventHandler, this);
		pair->second.SetConnection(&pair->first);

		// Connect to the server.
		auto result = pair->second.Open(url);

		return result.IsSuccess() ? 0 : FromPlatformError(ERROR_CANNOT_CONNECT);
	}

	int Send(void const* data, size_t size) {
		auto result = pair->second.SendBinary(data, size);
		return result.IsSuccess() ? 0 : -1;
	}

#ifdef __NX__
	bool Process() {
		if (pair->second.IsAvailable()) {
			pair->second.Perform();
			return true;
		}
		return false;
	}
#endif

private:
	std::unique_ptr<std::pair<nn::websocket::Connection, nn::websocket::WebSocket>> pair;
	bool isOpen = false;

	template<typename FN>
	void SafelyInvoke(FN fn) {
		try {
			fn();
		} catch (std::exception const& ex) {
			DebugWriteLine(_T("[WebSocketImpl::SafelyInvoke] function threw an exception \"%s\""), ex.what());
			assert(false);
		}
	}

	// Handle server connection events.
	static void OpenEventHandler(void* p) noexcept {
		reinterpret_cast<WebSocketImpl*>(p)->isOpen = true;
	}

	// Handle message receipt events.
	static void MessageEventHandler(void* pBuffer, size_t size, bool isText, void* p) noexcept {
		if (isText) {
			NN_LOG("TxtFrame: %.*s\n", static_cast<int>(size), reinterpret_cast<const char*>(pBuffer));
		} else {
			NN_LOG("BinFrame: size = %zu\n", size);
		}
		reinterpret_cast<WebSocketImpl*>(p)->OnReceived(std::string_view(reinterpret_cast<char*>(pBuffer), size));
	}

	// Handle server disconnection events.
	static void CloseEventHandler(std::uint16_t code, const char* pReason, size_t length, void* p) noexcept {
		UNREFERENCED_PARAMETER(code);
		UNREFERENCED_PARAMETER(pReason);
		UNREFERENCED_PARAMETER(length);
		NN_LOG("Close: code = %u, reason = %.*s\n", code, static_cast<int>(length), pReason);
		reinterpret_cast<WebSocketImpl*>(p)->isOpen = false;
		reinterpret_cast<WebSocketImpl*>(p)->OnClosed();
	}

	// Handle error events.
	static void ErrorEventHandler(nn::Result result, void*) noexcept {
		// TODO:  handle errors on Open.
		UNREFERENCED_PARAMETER(result);
		NN_LOG("Error: 0x%08X\n", result.GetInnerValueForDebug());
	}
};

#include "WebSocket.inl"
