#include "pch.h"
#include <net.h>
#include <libhttp.h>
#include <HttpUtil.h>
#include <Http.h>

#pragma comment(lib, "libSceHttp_stub_weak.a")
#pragma comment(lib, "libSceNet_stub_weak.a")
#pragma comment(lib, "libSceSsl_stub_weak.a")

using namespace Twitch;

namespace {
	int Initialize() {
		auto result = sceNetPoolCreate("Twitch net pool", 64 * 1024, 0);
		if (result >= 0) {
			auto const netMemoryId = result;
			result = sceSslInit(384 * 1024);
			if (result >= 0) {
				auto const sslId = result;
				result = sceHttpInit(netMemoryId, sslId, 64 * 1024);
				if (result >= 0) {
					return result;
				}
				sceSslTerm(sslId);
			}
			sceNetPoolDestroy(netMemoryId);
		}
		return result;
	}

	struct Internets {
		int Template;
		int Connection;
		int Request;
		Internets() : Template(0), Connection(0), Request(0) {}
		~Internets() {
			sceHttpDeleteRequest(Request);
			sceHttpDeleteConnection(Connection);
			sceHttpDeleteTemplate(Template);
		}
		static int GetHttpRequest(uintptr_t handle) {
			auto const* p = reinterpret_cast<Internets*>(handle);
			return p->Request;
		}
	};
}

#define ReturnIfFailedT(result,type) do { auto result_= (result); if(result_ < 0) return std::make_tuple(result_, type()); } while(false)
#define ReturnIfFailed(result) ReturnIfFailedT((result), uintptr_t)

std::tuple<int, uintptr_t> Platform::StartHttpRequest(string_t url, HttpRequestType requestType,
	std::chrono::milliseconds timeout, std::vector<HttpParam> const& requestHeaders, std::vector<char> const& requestBody) {
	// Initialize the HTTP library.  This occurs once since it's static.
	static auto const httpId = Initialize();
	ReturnIfFailed(httpId);

	// Create an HTTP template.
	auto httpTemplate = sceHttpCreateTemplate(httpId, "Twitch in Games SDK", SCE_HTTP_VERSION_1_1, SCE_TRUE);
	ReturnIfFailed(httpTemplate);
	auto internets = std::make_unique<Internets>();
	internets->Template = httpTemplate;

	// Set all time-outs to the requested value.
	auto const timeoutUs = static_cast<std::uint32_t>(1000 * timeout.count());
	ReturnIfFailed(sceHttpSetConnectTimeOut(httpTemplate, timeoutUs));
	ReturnIfFailed(sceHttpSetSendTimeOut(httpTemplate, timeoutUs));
	ReturnIfFailed(sceHttpSetRecvTimeOut(httpTemplate, timeoutUs));

	// Create the HTTP connection.
	auto httpConnection = sceHttpCreateConnectionWithURL(httpTemplate, url, SCE_TRUE);
	ReturnIfFailed(httpConnection);
	internets->Connection = httpConnection;

	// Determine the HTTP verb.
	int method;
	switch (requestType) {
	case HttpRequestType::HTTP_PUT_REQUEST:
		method = SCE_HTTP_METHOD_PUT;
		break;
	case HttpRequestType::HTTP_POST_REQUEST:
		method = SCE_HTTP_METHOD_POST;
		break;
	case HttpRequestType::HTTP_DELETE_REQUEST:
		method = SCE_HTTP_METHOD_DELETE;
		break;
	default:
		method = SCE_HTTP_METHOD_GET;
		break;
	}

	// Create the HTTP request.
	auto httpRequest = sceHttpCreateRequestWithURL(httpConnection, method, url, 0);
	ReturnIfFailed(httpRequest);
	internets->Request = httpRequest;

	// Add all request headers with replacement.
	for (auto const& httpParam : requestHeaders) {
		sceHttpAddRequestHeader(httpRequest, httpParam.paramName.c_str(), httpParam.paramValue.c_str(), SCE_HTTP_HEADER_OVERWRITE);
	}

	// Send the HTTP request.
	ReturnIfFailed(sceHttpSendRequest(httpRequest, requestBody.empty() ? nullptr : requestBody.data(), requestBody.size()));

	return std::make_tuple(0, reinterpret_cast<uintptr_t>(internets.release()));
}

int Platform::GetHttpResponseResultCode(uintptr_t handle) {
	// Extract the status code from the response.
	int statusCode;
	int const result = sceHttpGetStatusCode(Internets::GetHttpRequest(handle), &statusCode);
	return result < 0 ? result : statusCode;
}

#undef ReturnIfFailed
#define T std::map<tstring, tstring>
#define ReturnIfFailed(result) ReturnIfFailedT((result), T)

std::tuple<int, std::map<tstring, tstring>> Platform::GetHttpResponseHeaders(uintptr_t handle) {
	// Extract the headers.
	char* headerBuffer;
	size_t headersSize;
	sceHttpGetAllResponseHeaders(Internets::GetHttpRequest(handle), &headerBuffer, &headersSize);

	// Parse the headers into a dictionary.
	std::map<std::string, std::string> responseHeaders;
	ParseHeaders(headerBuffer, headersSize, responseHeaders);
	return std::make_tuple(0, responseHeaders);
}

#undef ReturnIfFailed
#define ReturnIfFailed(result) ReturnIfFailedT((result), std::vector<char>)

std::tuple<int, std::vector<char>> Platform::GetHttpResponse(uintptr_t handle, std::chrono::milliseconds /*timeout*/) {
	// Read the HTTP response and invoke the call-back.
	auto const httpRequest = Internets::GetHttpRequest(handle);
	char buffer[1024];
	std::vector<char> response;
	for (;;) {
		auto result = sceHttpReadData(httpRequest, buffer, sizeof(buffer));
		ReturnIfFailed(result);
		if (result == 0) {
			break;
		} else {
			response.insert(response.cend(), buffer, buffer + result);
		}
	}
	return std::make_tuple(0, response);
}

void Platform::FinishHttpRequest(uintptr_t& handle, std::chrono::milliseconds /*timeout*/) {
	auto handle_ = __sync_swap(&handle, std::remove_reference_t<decltype(handle)>());
	std::unique_ptr<Internets>(reinterpret_cast<Internets*>(handle_));
}
