#include "pch.h"
#include <config.h>
#include <mbedtls/net_sockets.h>
#include <mbedtls/entropy.h>
#include <mbedtls/ctr_drbg.h>
#include <mbedtls/certs.h>
#include "SecureSocket.h"

using namespace Twitch;

namespace {
	constexpr char const secureSocketName[] = "Twitch::SecureSocket::SecureSocketImpl";
}

struct SecureSocket::SecureSocketImpl {
	mbedtls_net_context net{};
	mbedtls_entropy_context entropy{};
	mbedtls_ctr_drbg_context ctr_drbg{};
	mbedtls_ssl_context ssl{};
	mbedtls_ssl_config config{};
	mbedtls_x509_crt cacert{};

	SecureSocketImpl() {
		int errorCode;
		mbedtls_net_init(&net);
		mbedtls_ssl_init(&ssl);
		mbedtls_ssl_config_init(&config);
		mbedtls_x509_crt_init(&cacert);
		mbedtls_ctr_drbg_init(&ctr_drbg);
		mbedtls_entropy_init(&entropy);
		errorCode = mbedtls_ctr_drbg_seed(&ctr_drbg, mbedtls_entropy_func, &entropy,
			reinterpret_cast<unsigned char const*>(secureSocketName),
			sizeof(secureSocketName));
		if(errorCode != 0) {
			DebugWriteLine("[SecureSocket::SecureSocketImpl::SecureSocketImpl] mbedtls_ctr_drbg_seed failed:  %d", errorCode);
			throw TwitchException(errorCode);
		}
		errorCode = mbedtls_x509_crt_parse(&cacert, reinterpret_cast<unsigned char const*>(mbedtls_test_cas_pem),
			mbedtls_test_cas_pem_len);
		if(errorCode < 0) {
			DebugWriteLine("[SecureSocket::SecureSocketImpl::SecureSocketImpl] mbedtls_x509_crt_parse failed:  %d", errorCode);
			throw TwitchException(errorCode);
		}
	}

	SecureSocketImpl(SecureSocketImpl const&) = delete;
	SecureSocketImpl(SecureSocketImpl&& that) = delete;
	SecureSocketImpl& operator=(SecureSocketImpl const&) = delete;
	SecureSocketImpl& operator=(SecureSocketImpl&& that) = delete;

	~SecureSocketImpl() {
		mbedtls_ssl_close_notify(&ssl);
		mbedtls_net_free(&net);
		mbedtls_x509_crt_free(&cacert);
		mbedtls_ssl_free(&ssl);
		mbedtls_ssl_config_free(&config);
		mbedtls_ctr_drbg_free(&ctr_drbg);
		mbedtls_entropy_free(&entropy);
	}

	void Close() {
		sceNetSocketAbort(net.fd, 0);
		mbedtls_net_free(&net);
	}

	int Connect(string_t hostName, unsigned short port, std::chrono::milliseconds timeout) noexcept {
		int errorCode = mbedtls_net_connect(&net, hostName, std::to_string(port).c_str(), timeout.count());
		if(errorCode < 0) {
			return errorCode;
		}
		errorCode = mbedtls_ssl_config_defaults(&config, MBEDTLS_SSL_IS_CLIENT,
			MBEDTLS_SSL_TRANSPORT_STREAM, MBEDTLS_SSL_PRESET_DEFAULT);
		if(errorCode < 0) {
			return errorCode;
		}
		mbedtls_ssl_conf_authmode(&config, MBEDTLS_SSL_VERIFY_OPTIONAL); // TODO:  https://ps4.siedev.net/resources/documents/SDK/6.500/Ssl-Overview/0001.html#__document_toc_00000004
		mbedtls_ssl_conf_ca_chain(&config, &cacert, nullptr);
		mbedtls_ssl_conf_rng(&config, mbedtls_ctr_drbg_random, &ctr_drbg);
		errorCode = mbedtls_ssl_setup(&ssl, &config);
		if(errorCode < 0) {
			return errorCode;
		}
		errorCode = mbedtls_ssl_set_hostname(&ssl, hostName);
		if(errorCode < 0) {
			return errorCode;
		}
		mbedtls_ssl_set_bio(&ssl, &net, mbedtls_net_send, mbedtls_net_recv, nullptr);
		while(errorCode = mbedtls_ssl_handshake(&ssl), errorCode != 0) {
			if(errorCode != MBEDTLS_ERR_SSL_WANT_READ && errorCode != MBEDTLS_ERR_SSL_WANT_WRITE) {
				DebugWriteLine("[SecureSocket::SecureSocketImpl::Connect] mbedtls_ssl_handshake failed:  %#x", errorCode);
				return errorCode;
			}
		}
		return 0;
	}

	int Receive(void* buffer, size_t size) noexcept {
		return mbedtls_ssl_read(&ssl, reinterpret_cast<unsigned char*>(buffer), size);
	}

	int Send(void const* buffer, size_t size) noexcept {
		return mbedtls_ssl_write(&ssl, reinterpret_cast<unsigned char const*>(buffer), size);
	}
};

SecureSocket::SecureSocket() {
	pimpl = std::make_unique<SecureSocket::SecureSocketImpl>();
}

SecureSocket::SecureSocket(SecureSocket&& that) : Socket(std::move(that)) {
	std::swap(pimpl, that.pimpl);
}

SecureSocket::~SecureSocket() = default;

void SecureSocket::InternalClose() noexcept {
	return pimpl->Close();
}

int Twitch::SecureSocket::InternalConnect(string_t hostName, unsigned short port, std::chrono::milliseconds timeout) noexcept {
	return pimpl->Connect(hostName, port, timeout);
}

int Twitch::SecureSocket::InternalReceive(void* buffer, size_t size) noexcept {
	return pimpl->Receive(buffer, size);
}

int Twitch::SecureSocket::InternalSend(void const* buffer, size_t size) noexcept {
	return pimpl->Send(buffer, size);
}
