using namespace std::string_literals;
using namespace Twitch;

extern char const* const sdkVersion;

namespace {
	using namespace ttv;

	struct ScienceEvent {
		using vector = std::vector<ScienceEvent>;

		tstring clientId, token, userId, method;
		stringmap properties;
	};

	char const* const sdkPlatform = PLATFORM;
	std::atomic<ScienceEvent::vector*> events;
	std::promise<void> stop_event;
	tstring const spadeUrl = _T("https://spade.twitch.tv/");
	std::string applicationName, deviceId;
	std::future<void> event_task;
	tstring const empty;

	std::string EncodeProperties(stringmap const& properties) {
		json::Value jsonValue(json::ValueType::objectValue);
		for (auto const& property : properties) {
			auto key = FromTstring(property.first);
			auto value = FromTstring(property.second);
			jsonValue[key] = value;
		}
		json::FastWriter writer;
		return writer.write(jsonValue);
	}

	void SendScienceEvent(ScienceEvent const& scienceEvent) {
		// Get the user identifier, if available or necessary.
		std::string userId;
		if (!scienceEvent.userId.empty()) {
			userId = FromTstring(scienceEvent.userId);
		} else if (!scienceEvent.token.empty()) {
			userId = "$token";
		}

		// Create a JSON payload for science.
		json::Value value(json::ValueType::objectValue);
		value["event"] = "sdk_request";
		json::Value properties(json::ValueType::objectValue);
		properties["client_id"] = FromTstring(scienceEvent.clientId);
		properties["sdk_method"] = FromTstring(scienceEvent.method);
		properties["user_id"] = userId;
		properties["sdk_version"] = sdkVersion;
		properties["platform"] = sdkPlatform;
		properties["runtime"] = sdkRuntime;
		properties["application_name"] = applicationName;
		properties["device_id"] = deviceId;
		properties["sdk_json_payload"] = EncodeProperties(scienceEvent.properties);
		value["properties"] = properties;

		// Format the payload as a base64-encoded POST request and send it to Spade.
		json::FastWriter writer;
		auto s = writer.write(value);
		auto data = Base64Encode(std::vector<char>(s.cbegin(), s.cend()));
		auto const httpResponse = HttpRequest().Post(spadeUrl, std::vector<char>(data.cbegin(), data.cend()));
#ifdef _DEBUG
		int const result = httpResponse.Result;
		if (result == 0) {
			auto const resultCode = httpResponse.ResultCode;
			auto const responseBody = httpResponse.Response;
			UNREFERENCED_PARAMETER(resultCode);
			UNREFERENCED_PARAMETER(responseBody);
		}
#else
		UNREFERENCED_PARAMETER(httpResponse);
#endif
	}

	void SendScienceEvents() {
		auto const check_stop = stop_event.get_future();
		do {
			std::unique_ptr<ScienceEvent::vector> p(events.exchange(nullptr));
			if (p) {
				for (auto const& e : *p) {
					try {
						SendScienceEvent(e);
					} catch (std::exception const& ex) {
						DebugWriteLine(_T("SendScienceEvents failed: %s"), ToTstring(ex.what()).c_str());
						UNREFERENCED_PARAMETER(ex);
					}
				}
			}
		} while (check_stop.wait_for(std::chrono::seconds(5)) != std::future_status::ready);
}

	void StopScienceEvents() {
		stop_event.set_value();
	}

	void SendScienceEvent(tstring const& clientId, tstring const& token, tstring const& userId, tstring const& method, stringmap const& properties) noexcept {
		try {
			std::unique_ptr<ScienceEvent::vector> p(events.exchange(nullptr));
			if (p == nullptr) {
				p.reset(new ScienceEvent::vector);
			}
			p->push_back({ clientId, token, userId, method, properties });
			events.exchange(p.release());
			if (!event_task.valid()) {
				applicationName = GetApplicationName();
				deviceId = GetDeviceId();
				event_task = std::async(std::launch::async, SendScienceEvents);
				atexit(StopScienceEvents);
			}
		} catch (std::exception const& ex) {
			DebugWriteLine(_T("warning: SendScienceEvent failed: %s"), ToTstring(ex.what()).c_str());
			UNREFERENCED_PARAMETER(ex);
			// Ignore exceptions.
		} catch (...) {
			DebugWriteLine(_T("warning: SendScienceEvent failed"));
			// Ignore exceptions.
		}
	}
}

void Twitch::SendTokenScienceEvent(tstring const& clientId, tstring const& token, tstring const& method, stringmap const& properties) noexcept {
	SendScienceEvent(clientId, token, empty, method, properties);
}

void Twitch::SendUserScienceEvent(tstring const& clientId, tstring const& userId, tstring const& method, stringmap const& properties) noexcept {
	SendScienceEvent(clientId, empty, userId, method, properties);
}
