#include "Http.h"
#include "pch.h"
#include "StringUtil.h"
#include "HttpUtil.h"
#include "Http.h"

using namespace Twitch;

namespace {
	string_t const helixAuthorizationPrefix = _T("Bearer ");
	string_t const krakenAuthorizationPrefix = _T("OAuth ");
	string_t const clientIdHeaderName = _T("Client-ID");

	bool IsKrakenUrl(tstring const& url) {
		return url.find(_T("https://api.twitch.tv/kraken/")) == 0;
	}
}

constexpr std::chrono::seconds HttpRequest::DefaultTimeout /*= std::chrono::seconds(10)*/;

HttpParam::HttpParam(tstring const& name, tstring const& value)
	: paramName(name)
	, paramValue(value) {}

HttpParam::HttpParam(tstring const& name, int value)
	: paramName(name)
	, paramValue(to_tstring(value)) {}

bool Twitch::IsTwitchEndpoint(string_t url) {
	Uri uri(url);
	tstring const host = ToLowerCase(uri.GetHostName());
	return host == _T("twitch.tv") || EndsWith(host, _T(".twitch.tv"));
}

std::vector<char> const HttpRequest::EmptyBody;
tstring const HttpRequest::EmptyString;

HttpRequest::HttpRequest(tstring const& clientId/*= EmptyString*/, tstring const& token/*= EmptyString*/) :
	clientId(clientId),
	token(token),
	timeout(DefaultTimeout) {}

HttpRequest::~HttpRequest() {}

HttpResponse HttpRequest::Get(tstring const& url) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HttpRequestType::HTTP_GET_REQUEST, EmptyBody);
	return HttpResponse(handle, timeout, result);
}

HttpResponse HttpRequest::Post(tstring const& url, std::vector<char> const& requestBody, bool isFormUrlEncoded /*= false*/) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HttpRequestType::HTTP_POST_REQUEST, requestBody, isFormUrlEncoded);
	return HttpResponse(handle, timeout, result);
}

HttpResponse HttpRequest::Put(tstring const& url, std::vector<char> const& requestBody, bool isFormUrlEncoded /*= false*/) {
	int result;
	uintptr_t handle;
	std::tie(result, handle) = StartHttpRequest(url, HttpRequestType::HTTP_PUT_REQUEST, requestBody, isFormUrlEncoded);
	return HttpResponse(handle, timeout, result);
}

std::tuple<int, uintptr_t> HttpRequest::StartHttpRequest(tstring const& url, HttpRequestType requestType, std::vector<char> const& requestBody, bool isFormUrlEncoded /*= false*/) {
	// Validate the arguments.
	if (url.empty()) {
		return std::make_tuple(FromPlatformError(ERROR_BAD_ARGUMENTS), 0);
	}
	switch (requestType) {
	case HttpRequestType::HTTP_GET_REQUEST:
	case HttpRequestType::HTTP_PUT_REQUEST:
	case HttpRequestType::HTTP_POST_REQUEST:
	case HttpRequestType::HTTP_DELETE_REQUEST:
		break;
	default:
		return std::make_tuple(FromPlatformError(ERROR_BAD_ARGUMENTS), 0);
	}

	// Add request headers.
	std::vector<HttpParam> requestHeaders;

	// Add the Kraken Accept header, if necessary.
	bool isKrakenUrl = IsKrakenUrl(url);
	if (isKrakenUrl) {
		requestHeaders.emplace_back(HttpParam(_T("Accept"), _T("application/vnd.twitchtv.v5+json")));
	}

	// Add the authorization token, if provided.
	if (!token.empty()) {
		auto authorizationPrefix = isKrakenUrl ? krakenAuthorizationPrefix : helixAuthorizationPrefix;
		requestHeaders.emplace_back(HttpParam(_T("Authorization"), authorizationPrefix + token));
	}

#ifdef _WIN32
	// Prefer to receive gzipped responses if no other specified.
	requestHeaders.emplace_back(HttpParam(_T("Accept-Encoding"), _T("gzip")));
#endif

	// Add content headers.
	if (!requestBody.empty()) {
		requestHeaders.emplace_back(HttpParam(_T("Content-Length"), to_tstring(requestBody.size())));
		if (isFormUrlEncoded) {
			requestHeaders.emplace_back(HttpParam(_T("Content-Type"), _T("application/x-www-form-urlencoded")));
		}
	}

	// Add the client ID, if appropriate.
	if (!clientId.empty() && IsTwitchEndpoint(url.c_str())) {
		requestHeaders.emplace_back(HttpParam(clientIdHeaderName, clientId));
	}

	return Platform::StartHttpRequest(url.c_str(), requestType, timeout, requestHeaders, requestBody);
}

HttpResponse::HttpResponse(uintptr_t handle, std::chrono::milliseconds timeout, int result) : handle(handle), timeout(timeout), result(result) {}

HttpResponse::~HttpResponse() {
	Platform::FinishHttpRequest(handle, timeout);
}

stringmap const& HttpResponse::GetHeaders() const {
	if (headers.empty()) {
		int result_;
		std::tie(result_, const_cast<HttpResponse*>(this)->headers) = Platform::GetHttpResponseHeaders(handle);
		UNREFERENCED_PARAMETER(result_); // TODO
	}
	return headers;
}

std::vector<char> const& HttpResponse::GetResponse() const {
	if (response.empty()) {
		int result_;
		std::tie(result_, const_cast<HttpResponse*>(this)->response) = Platform::GetHttpResponse(handle, timeout);
		UNREFERENCED_PARAMETER(result_); // TODO
	}
	return response;
}

int HttpResponse::GetResultCode() const {
	if (resultCode == 0) {
		const_cast<HttpResponse*>(this)->resultCode = Platform::GetHttpResponseResultCode(handle);
	}
	return resultCode;
}

Twitch::HttpResponse::HttpResponse(HttpResponse&& that) : headers(std::move(that.headers)), response(std::move(that.response)), handle(that.handle), timeout(that.timeout), resultCode(that.resultCode), result(that.result) {
	that.handle = 0;
}

WebSocket::ClosedFn const WebSocket::DefaultClosedFn = [] {};
WebSocket::ReceivedFn const WebSocket::DefaultReceivedFn = [](auto) {};
