#pragma once

namespace Twitch {
	int const defaultTimeoutInSeconds = 10;

	enum class HttpRequestType {
		HTTP_INVALID_REQUEST = -1,
		HTTP_GET_REQUEST,
		HTTP_PUT_REQUEST,
		HTTP_POST_REQUEST,
		HTTP_DELETE_REQUEST,
		HTTP_WEB_SOCKET_REQUEST,
		HTTP_REQUEST_COUNT
	};

	struct HttpParam {
		HttpParam(tstring const& name, tstring const& value);
		HttpParam(tstring const& name, int value);

		tstring paramName;
		tstring paramValue;
	};

	typedef std::function<bool(unsigned statusCode, stringmap const& headers, void* userData)> HttpRequestHeadersCallback;
	typedef std::function<void(unsigned statusCode, std::vector<char> const& body, void* userData)> HttpRequestCallback;

	// Determines if the given url represents a Twitch endpoint.
	bool IsTwitchEndpoint(string_t url);

	namespace Platform {
		void FinishHttpRequest(uintptr_t& handle, std::chrono::milliseconds timeout);
		std::tuple<int, std::vector<char>> GetHttpResponse(uintptr_t handle, std::chrono::milliseconds timeout);
		std::tuple<int, stringmap> GetHttpResponseHeaders(uintptr_t handle);
		int GetHttpResponseResultCode(uintptr_t handle);
		std::tuple<int, uintptr_t> StartHttpRequest(string_t url, HttpRequestType requestType, std::chrono::milliseconds timeout, std::vector<HttpParam> const& requestHeaders, std::vector<char> const& requestBody);
	}

	class HttpResponse {
	public:
		~HttpResponse();
		int GetResult() const { return result; }
		stringmap const& GetHeaders() const;
		std::vector<char> const& GetResponse() const;
		int GetResultCode() const;
		static bool IsSuccessful(unsigned statusCode) { return statusCode >= 200 && statusCode < 300; }

		__declspec(property(get = GetResult)) int const Result;
		__declspec(property(get = GetHeaders)) stringmap const& Headers;
		__declspec(property(get = GetResponse)) std::vector<char> const& Response;
		__declspec(property(get = GetResultCode)) int const ResultCode;

		HttpResponse() = delete;
		HttpResponse(HttpResponse const&) = delete;
		HttpResponse(HttpResponse&&);
		HttpResponse& operator=(HttpResponse const&) = delete;

	private:
		friend class HttpRequest;

		stringmap headers;
		std::vector<char> response;
		uintptr_t handle;
		std::chrono::milliseconds timeout;
		int resultCode{ 0 };
		int result;

		HttpResponse(uintptr_t handle, std::chrono::milliseconds timeout, int result);
	};

	class HttpRequest {
	public:
		HttpRequest(tstring const& clientId = EmptyString, tstring const& token = EmptyString);
		~HttpRequest();

		HttpResponse Get(tstring const& url);
		HttpResponse Post(tstring const& url, std::vector<char> const& requestBody = EmptyBody, bool isFormUrlEncoded = false);
		HttpResponse Put(tstring const& url, std::vector<char> const& requestBody = EmptyBody, bool isFormUrlEncoded = false);
		tstring const& GetClientId() const { return clientId; }
		void SetClientId(tstring const& clientId_) { clientId = clientId_; }
		tstring const& GetToken() const { return token; }
		void SetToken(tstring const& token_) { token = token_; }
		std::chrono::milliseconds GetTimeout() const { return timeout; }
		void SetTimeout(std::chrono::milliseconds timeout_) { timeout = timeout_; }

		__declspec(property(get = GetClientId, put = SetClientId)) tstring const& ClientId;
		__declspec(property(get = GetToken, put = SetToken)) tstring const& Token;
		__declspec(property(get = GetTimeout, put = SetTimeout)) std::chrono::milliseconds const Timeout;

		static std::vector<char> const EmptyBody;
		static tstring const EmptyString;
		static constexpr std::chrono::seconds DefaultTimeout = std::chrono::seconds(10);

	private:
		tstring clientId;
		tstring token;
		std::chrono::milliseconds timeout;

		std::tuple<int, uintptr_t> StartHttpRequest(tstring const& url, HttpRequestType requestType,
			std::vector<char> const& requestBody, bool isFormUrlEncoded = false);
	};

	class WebSocket {
	public:
		using ClosedFn = std::function<void()>;
		using ReceivedFn = std::function<void(std::string_view)>;

		WebSocket();
		WebSocket(WebSocket const&) = delete;
		WebSocket(WebSocket&&) noexcept;
		WebSocket& operator=(WebSocket const&) = delete;
		WebSocket& operator=(WebSocket&&) noexcept;
		~WebSocket();

#ifdef __NX__
		bool Process();
#endif
		int Open(string_t url, std::chrono::milliseconds timeout = HttpRequest::DefaultTimeout);
		void Close();
		int Send(void const* data, size_t size);
		template<typename T>
		int Send(T const& data) { return Send(data.data(), data.size()); }
		bool GetIsOpen() const;
		__declspec(property(get = GetIsOpen)) bool const IsOpen;
		ClosedFn GetOnClosed() const;
		void SetOnClosed(ClosedFn closedFn);
		__declspec(property(get = GetOnClosed, put = SetOnClosed)) ClosedFn const OnClosed;
		ReceivedFn GetOnReceived() const;
		void SetOnReceived(ReceivedFn receivedFn);
		__declspec(property(get = GetOnReceived, put = SetOnReceived)) ReceivedFn const OnReceived;

		static ClosedFn const DefaultClosedFn /*= [] {}*/;
		static ReceivedFn const DefaultReceivedFn /*= [](auto) {}*/;

	private:
		struct WebSocketImpl;

		std::unique_ptr<WebSocketImpl> pimpl;
	};

#if defined(_WIN32)
	struct Internets {
		std::vector<HINTERNET> handles;
		std::vector<char> buffer;
		std::promise<int> sendPromise;
		std::promise<int> responsePromise;
		std::promise<int> dataPromise;
		std::promise<void> shutdownPromise;
		std::promise<void> closePromise;
		std::function<void(char*, size_t)> onReceive;

		~Internets();
		void Add(HINTERNET internet);
		void ReplaceHttpRequest(HINTERNET internet) noexcept;
		void Close() noexcept;
		void OnClose(HINTERNET hInternet) noexcept;
		void EndResponse(HINTERNET hInternet, DWORD errorCode = 0) noexcept;
		void EndSend(HINTERNET hInternet, DWORD errorCode = 0) noexcept;
		void EndData(HINTERNET hInternet, DWORD errorCode = 0) noexcept;
		static HINTERNET GetHttpRequest(uintptr_t handle) noexcept;
		static VOID CALLBACK WebSocketCallback(HINTERNET hInternet, DWORD_PTR dwContext, DWORD dwInternetStatus, LPVOID lpvStatusInformation, DWORD dwStatusInformationLength) noexcept;
		static VOID CALLBACK CloseCallback(HINTERNET hInternet, DWORD_PTR dwContext, DWORD dwInternetStatus, LPVOID lpvStatusInformation, DWORD dwStatusInformationLength) noexcept;
		static VOID CALLBACK DataCallback(HINTERNET hInternet, DWORD_PTR dwContext, DWORD dwInternetStatus, LPVOID lpvStatusInformation, DWORD dwStatusInformationLength) noexcept;
		static VOID CALLBACK ResponseCallback(HINTERNET hInternet, DWORD_PTR dwContext, DWORD dwInternetStatus, LPVOID lpvStatusInformation, DWORD dwStatusInformationLength) noexcept;
		static VOID CALLBACK SendCallback(HINTERNET hInternet, DWORD_PTR dwContext, DWORD dwInternetStatus, LPVOID lpvStatusInformation, DWORD dwStatusInformationLength) noexcept;
	};
#endif
}
