#include "pch.h"
#include "StringUtil.h"
#include "HttpUtil.h"

using namespace Twitch;

namespace {
	std::string const safeCharacters = "!'()*,-.:;@_~";

	bool NeedsEncoding(char ch) {
		if(safeCharacters.find(ch) != safeCharacters.npos) {
			return false;
		}
		return !isalnum(static_cast<unsigned char>(ch));
	}
}


void Twitch::UrlEncode(tstring const& inputString, tstringstream& outputStream) {
	outputStream << std::hex << std::uppercase;
	for(char currentCharacter : FromTstring(inputString)) {
		if(NeedsEncoding(currentCharacter)) {
			outputStream << "%" << static_cast<unsigned>(static_cast<unsigned char>(currentCharacter));
		} else {
			outputStream << currentCharacter;
		}
	}
}


tstring Twitch::UrlEncode(tstring const& inputString) {
	tstringstream stream;
	UrlEncode(inputString, stream);
	return stream.str();
}


void Twitch::UrlDecode(tstring const& inputString, tstringstream& outputStream) {
	outputStream << std::hex << std::uppercase;
	size_t index = 0;
	while(index < inputString.size()) {
		if(inputString[index] == _T('%')) {
			// Convert the next 2 hex digits into a character
			if(inputString.size() > index + 2) {
				TCHAR hexValue[4];
				hexValue[0] = inputString.data()[index + 1];
				hexValue[1] = inputString.data()[index + 2];
				hexValue[2] = _T('\0');

				int intValue = 0;
				_stscanf_s(hexValue, _T("%x"), &intValue);

				outputStream << static_cast<TCHAR>(intValue);

				index += 3;
			}
			// Invalid encoding
			else {
				return;
			}
		} else {
			outputStream << inputString[index];
			index++;
		}
	}
}


void Twitch::UrlDecode(tstring const& input, tstring& result) {
	tstringstream sstream;
	UrlDecode(input, sstream);
	result = sstream.str();
}


tstring Twitch::UrlDecode(tstring const& inputString) {
	tstringstream sstream;
	UrlDecode(inputString, sstream);
	return sstream.str();
}


tstring Twitch::BuildUrlEncodedRequestParams(std::vector<std::pair<tstring, tstring>> const& requestParams) {
	tstringstream paramsStream;
	for(auto it = requestParams.begin(); it != requestParams.end(); ++it) {
		if(it != requestParams.begin()) {
			paramsStream << _T("&");
		}
		UrlEncode(it->first, paramsStream);
		paramsStream << _T("=");
		UrlEncode(it->second, paramsStream);
	}
	return paramsStream.str();
}


tstring Twitch::BuildUrlEncodedRequestParams(std::vector<HttpParam> const& requestParams) {
	tstringstream paramsStream;
	for(auto it = requestParams.begin(); it != requestParams.end(); ++it) {
		if(it != requestParams.begin()) {
			paramsStream << _T("&");
		}
		UrlEncode(it->paramName, paramsStream);
		paramsStream << _T("=");
		UrlEncode(it->paramValue, paramsStream);
	}
	return paramsStream.str();
}


tstring Twitch::BuildUrlEncodedRequestParams(stringmap const& requestParams) {
	tstringstream paramsStream;
	for(auto it = requestParams.begin(); it != requestParams.end(); ++it) {
		if(it != requestParams.begin()) {
			paramsStream << _T("&");
		}
		UrlEncode(it->first, paramsStream);
		paramsStream << _T("=");
		UrlEncode(it->second, paramsStream);
	}
	return paramsStream.str();
}


tstring Twitch::BuildHttpHeader(std::vector<HttpParam> const& headerParams) {
	tstringstream headerStream;

	for(auto it = headerParams.begin(); it != headerParams.end(); ++it) {
		headerStream << it->paramName.c_str();
		headerStream << _T(": ");
		headerStream << it->paramValue.c_str();
		headerStream << _T("\r\n");
	}

	return headerStream.str();
}


bool Twitch::ContainsHttpParameter(std::vector<HttpParam> const& headers, tstring const& name) {
	tstring lowerName = ToLowerCase(name);
	auto const iter = std::find_if(headers.begin(), headers.end(), [&lowerName](HttpParam const& p)->bool {
		tstring lowerParam = ToLowerCase(p.paramName);
		return lowerParam == lowerName;
	});
	return iter != headers.end();
}


int Twitch::SplitHttpParameters(tstring const& parameterString, std::vector<std::pair<tstring, tstring>>& result) {
	size_t currentIndex = 0;

	while(true) {
		auto const equalsIndex = parameterString.find(_T('='), currentIndex);

		if(equalsIndex == tstring::npos) {
			break;
		} else {
			auto const valueIndex = equalsIndex + 1;
			tstring key;
			UrlDecode(parameterString.substr(currentIndex, equalsIndex - currentIndex), key);

			tstring value;
			auto const ampersandIndex = parameterString.find(_T('&'), valueIndex);

			// If we didn't find the ampersand, we just get a substring to the end of the string.
			auto const valueLength = (ampersandIndex == tstring::npos) ? tstring::npos : ampersandIndex - valueIndex;
			UrlDecode(parameterString.substr(equalsIndex + 1, valueLength), value);

			result.emplace_back(key, value);

			if((ampersandIndex == tstring::npos) || (ampersandIndex == parameterString.size())) {
				break;
			} else {
				currentIndex = ampersandIndex + 1;
			}
		}
	}

	return ERROR_SUCCESS;
}


int Twitch::SplitHttpParameters(tstring const& parameterString, stringmap& result) {
	std::vector<std::pair<tstring, tstring>> params;
	int const ec = SplitHttpParameters(parameterString, params);

	if(ec == ERROR_SUCCESS) {
		for(auto const& pair : params) {
			result[pair.first] = pair.second;
		}
	}

	return ec;
}


static void InternalParseHeaders(tstring&& converted, stringmap& headers) {
	// Break the string into multiple lines.
	TCHAR const lineDelimeter[] = _T("\r\n");
	static size_t const lineDelimeterLength = _countof(lineDelimeter) - 1;
	for(;;) {
		// If there are no more lines, finish.
		size_t index = converted.find(lineDelimeter);
		if(index == tstring::npos || index == 0) {
			break;
		}

		// Extract and remove the line.
		tstring line = converted.substr(0, index);
		converted = converted.substr(index + lineDelimeterLength);

		// If the current line is formatted as an HTTP header, break it up into
		// a name-value pair.
		index = line.find(_T(':'));
		if(index != tstring::npos) {
			tstring const name = line.substr(0, index);
			tstring value = line.substr(index + 1);

			// Trim whitespace from the start of the value.
			value.erase(value.begin(), std::find_if(value.begin(), value.end(), [](TCHAR ch) { return !IsWhitespace(ch); }));

			// Add it to the dictionary, replacing any previous value.
			headers[name] = value;
		}
	}
}


void Twitch::ParseHeaders(string_t const raw, stringmap& headers) {
	InternalParseHeaders(tstring(raw), headers);
}


void Twitch::ParseHeaders(string_t const raw, size_t size, stringmap& headers) {
	// Create a string for parsing convenience.
	InternalParseHeaders(tstring(raw, raw + size), headers);
}


Uri::Uri() {}


Uri::Uri(tstring const& url) {
	SetUrl(url);
}


void Uri::SetUrl(tstring const& url) {
	DisassembleUrl(url);
}


void Uri::DisassembleUrl(tstring const& url) {
	mParams.clear();

	// Find the protocol
	size_t protocolIndex = url.find(_T("://"));
	if(protocolIndex != tstring::npos) {
		mProtocol = url.substr(0, protocolIndex);
		protocolIndex += 3;
	} else {
		mProtocol = _T("");
		protocolIndex = 0;
	}

	// Find the host name
	size_t slashIndex = url.find(_T('/'), protocolIndex);
	size_t const paramsIndex = url.find(_T('?'));
	if(paramsIndex != tstring::npos) {
		queryString = url.substr(paramsIndex);
	}
	if(slashIndex != tstring::npos) {
		mHostName = url.substr(protocolIndex, slashIndex - protocolIndex);
		mPath = url.substr(slashIndex, paramsIndex - slashIndex);
	} else {
		// No path
		if(paramsIndex != tstring::npos) {
			mHostName = url.substr(protocolIndex, paramsIndex - protocolIndex);
		} else {
			mHostName = url.substr(protocolIndex);
		}
		mPath = _T("/");
	}

	// Find the port
	size_t const colonIndex = mHostName.find(_T(':'));
	if(colonIndex != tstring::npos) {
		mPort = mHostName.substr(colonIndex + 1);
		mHostName = mHostName.substr(0, colonIndex);
	} else {
		mPort = _T("");
	}

	// Find the query paramters
	if(paramsIndex != tstring::npos) {
		std::vector<tstring> params;
		Split(url.substr(paramsIndex + 1), params, _T('&'), true);

		for(auto const& str : params) {
			tstring key;
			tstring value;

			size_t const index = str.find(_T('='));

			if(index != tstring::npos) {
				key = str.substr(0, index);
				value = str.substr(index + 1);
			} else {
				key = str;
			}

			if(key != _T("")) {
				mParams[key] = UrlDecode(value);
			}
		}
	}
}


tstring Uri::AssembleUrl() const {
	tstringstream url;

	if(mProtocol != _T("")) {
		url << mProtocol << _T("://");
	}

	url << mHostName;

	if(mPort != _T("")) {
		url << _T(':') << mPort;
	}

	url << mPath;

	if(!mParams.empty()) {
		url << _T("?") + BuildUrlEncodedRequestParams(mParams);
	}

	return url.str();
}


Uri::operator tstring() const {
	return GetUrl();
}


bool Uri::operator==(Uri const& other) const {
	if(mProtocol != other.mProtocol) {
		return false;
	}

	if(mHostName != other.mHostName) {
		return false;
	}

	if(mPort != other.mPort) {
		return false;
	}

	if(mPath != other.mPath) {
		return false;
	}

	if(mParams.size() != other.mParams.size()) {
		return false;
	}

	for(auto const& param : mParams) {
		auto const& iter = other.mParams.find(param.first);
		if(iter == other.mParams.end() || iter->second != param.second) {
			return false;
		}
	}

	return true;
}


bool Uri::operator!=(Uri const& other) const {
	return !(*this == other);
}


bool Uri::GetPort(std::uint16_t& result) const {
	result = 0;

	if(mPort == _T("")) {
		return false;
	}

	unsigned n;
	int const numRead = _stscanf_s(mPort.c_str(), _T("%u"), &n);
	if(numRead == 1) {
		result = static_cast<std::remove_reference_t<decltype(result)>>(n);
	}
	return numRead == 1;
}


void Uri::GetPathComponents(std::vector<tstring>& result) const {
	Split(mPath, result, _T('/'), false);
}


void Uri::ClearParams() {
	mParams.clear();
}


void Uri::SetParam(tstring const& param, string_t value) {
	if(value != nullptr) {
		mParams[param] = tstring(value);
	}
}


void Uri::SetParam(tstring const& param, tstring const& value) {
	mParams[param] = value;
}


void Uri::SetParam(tstring const& param, std::uint32_t value) {
	mParams[param] = to_tstring(value);
}


void Uri::SetParam(tstring const& param, std::int32_t value) {
	mParams[param] = to_tstring(value);
}


void Uri::SetParam(tstring const& param, std::uint64_t value) {
	mParams[param] = to_tstring(value);
}


void Uri::SetParam(tstring const& param, std::int64_t value) {
	mParams[param] = to_tstring(value);
}


void Uri::SetParam(tstring const& param, bool value) {
	mParams[param] = value ? _T("true") : _T("false");
}


bool Uri::ContainsParam(tstring const& param) const {
	return mParams.find(param) != mParams.end();
}


tstring Uri::GetUrl() const {
	return AssembleUrl();
}
