#pragma once

#include "Twitch.h"
#include "twitchsdk/core/json/writer.h"
#include "Http.h"
#include "StringUtil.h"

#ifdef _DEBUG
using DebugFn = void(*)(LPCTSTR);

extern DebugFn test_OutputDebugString;
#endif
#ifndef _UNICODE
# define _istspace isspace
#endif

namespace Twitch {
#ifdef _DEBUG
	void DebugWriteLine_(string_t format, ...);
# define DebugWriteLine(...) DebugWriteLine_(__VA_ARGS__)
#else
# define DebugWriteLine(...) ((void)0)
#endif
	tstring ExtractToken(tstring const& s, string_t name = _T("token"));

	template<typename FN>
	auto LockedOperation(std::mutex& m, FN fn) {
		std::lock_guard<decltype(m)> lock(m);
		return fn();
	}

	inline void ThrowIfFailed(int result) {
		if(result < 0) {
			throw TwitchException(result);
		}
	}

	inline void ThrowIfResponseFailed(HttpResponse const& httpResponse) {
		if(httpResponse.Result) {
			throw TwitchException(httpResponse.Result);
		}
		if(!HttpResponse::IsSuccessful(httpResponse.ResultCode)) {
#ifdef _DEBUG
			auto response = httpResponse.Response;
			auto s = tstring(response.cbegin(), response.cend());
			DebugWriteLine(_T("%d:  %s"), httpResponse.ResultCode, s.c_str());
#endif
			throw TwitchException(httpResponse.ResultCode);
		}
	}

	template<typename R, typename T>
	struct Handle {
		Handle(T value, std::function<R(T)> destroyFn) : value(value), destroyFn(destroyFn) {}
		~Handle() { destroyFn(value); }
		operator T() const { return value; }

	private:
		T value;
		std::function<R(T)> destroyFn;
	};

#ifdef _WIN32
# if WINAPI_FAMILY_PARTITION(WINAPI_PARTITION_DESKTOP)
	struct RegKey {
		RegKey() : key(HKEY()) {}
		~RegKey() { RegCloseKey(key); }
		HKEY* operator&() { return &key; }
		operator HKEY() const { return key; }

	private:
		HKEY key;
	};
# endif
#endif

	inline char AsHexDigit(int value) noexcept {
		value &= 0x0f;
		return static_cast<char>(value < 10 ? value + '0' : value - 10 + 'A');
	}

	std::string Base64Encode(std::vector<char> const& data);
	std::time_t ParseTime(std::string const& s);
	void SendTokenScienceEvent(tstring const& clientId, tstring const& token, tstring const& method, stringmap const& properties) noexcept;
	void SendUserScienceEvent(tstring const& clientId, tstring const& userId, tstring const& method, stringmap const& properties) noexcept;

#if defined(__ORBIS__) || defined(__NX__)
	void rand_s(unsigned* p);
#endif
}
