#include "pch.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "Json.h"

void Twitch::Stringify(std::ostream& os, double d) {
	if (isinf(d)) {
		throw std::runtime_error("number is infinite");
	}
	std::streamsize precision = os.precision();
	// https://en.wikipedia.org/wiki/Double-precision_floating-point_format#IEEE_754_double-precision_binary_floating-point_format:_binary64
	os << std::setprecision(15) << d << std::setprecision(precision);
}

void Twitch::Stringify(std::ostream& os, char const* s) {
	if (s != nullptr) {
		os << '"';
		for (char ch; ch = *s, ch; ++s) {
			if (static_cast<unsigned char>(ch) < ' ') {
				if (ch == '\b') {
					os << "\\b";
				} else if (ch == '\f') {
					os << "\\f";
				} else if (ch == '\n') {
					os << "\\n";
				} else if (ch == '\r') {
					os << "\\r";
				} else if (ch == '\t') {
					os << "\\t";
				} else {
					os << "\\u00" << AsHexDigit(ch >> 4) << AsHexDigit(ch);
				}
			} else if (ch == '"' || ch == '\\') {
				os << '\\' << ch;
			} else {
				os << ch;
			}
		}
		os << '"';
	} else {
		Stringify(os, nullptr);
	}
}

#ifdef _UNICODE
void Twitch::Stringify(std::ostream& os, wchar_t const* s) {
	if (s != nullptr) {
		auto const result = Twitch::FromTstring(s);
		Stringify(os, result);
	} else {
		Stringify(os, nullptr);
	}
}
#endif

using namespace Twitch;

Json::Json(Json::array const& values) : value(ttv::json::ValueType::arrayValue) {
	for (auto const& item : values) {
		value.append(item.value);
	}
}

Json::Json(array&& values) : value(ttv::json::ValueType::arrayValue) {
	for (auto const& item : values) {
		value.append(item.value);
	}
}

Json::Json(object const& values) : value(ttv::json::ValueType::objectValue) {
	for (auto const& pair : values) {
		value[pair.first] = pair.second.value;
	}
}

Json::Json(object&& values) : value(ttv::json::ValueType::objectValue) {
	for (auto const& pair : values) {
		value[pair.first] = pair.second.value;
	}
}

Json Json::operator[](std::string const& s) const {
	return Json(&Value[s]);
}

std::string Twitch::Json::dump() const {
	ttv::json::FastWriter writer;
	return writer.write(Value);
}

Twitch::Json::array Twitch::Json::array_items() const {
	array rv;
	for (auto& item : Value) {
		rv.push_back(Json(&item));
	}
	return rv;
}

Twitch::Json::object Twitch::Json::object_items() const {
	object rv;
	auto const& v = Value;
	for (auto const& name : v.getMemberNames()) {
		rv.insert({ name, Json(const_cast<ttv::json::Value*>(&v[name])) });
	}
	return rv;
}

std::string Twitch::Json::string_value() const {
	return Value.isString() ? Value.asString() : "";
}

int Twitch::Json::int_value() const {
	return static_cast<int>(Value.asInt());
}

Json Twitch::Json::Parse(std::string_view s, std::string& error) {
	ttv::json::Reader reader;
	ttv::json::Value value;
	if (reader.parse(s.data(), s.data() + s.size(), value)) {
		return Json(std::move(value));
	}
	error = "JSON syntax error";
	return Json();
}
