#include "pch.h"
#include "StringUtil.h"

using namespace Twitch;

namespace {
	std::uint32_t CountDays(std::int32_t y, std::int32_t m, std::int32_t d) {
		if(m <= 2) {
			y--;
		}

		std::int32_t const era = (y >= 0 ? y : y - 399) / 400;
		std::int32_t const yoe = y - era * 400;                             // [0, 399]
		std::int32_t const doy = (153 * (m + (m > 2 ? -3 : 9)) + 2) / 5 + d - 1;  // [0, 365]
		std::int32_t const doe = yoe * 365 + yoe / 4 - yoe / 100 + doy;         // [0, 146096]

		return era * 146097 + doe - 719468;
	}

	bool RFC3339TimeToTimePoint(const std::string& str, std::chrono::system_clock::time_point& result) {
		// typedef std::chrono::duration<int, std::ratio_multiply<std::ratio<24>, std::chrono::hours::period>> days;

		using namespace std::chrono;

		std::istringstream is(str);
		const std::time_get<char>& tg = std::use_facet<std::time_get<char>>(is.getloc());

		try {
			std::uint32_t base = 0;
			std::uint32_t offset = 0;

			// Read YYYY:MM:DT:HH:MM:SS
			if(!is.eof()) {
				char const patternYMDHMS[] = "%Y-%m-%dT%H:%M:%S";

				std::tm baseTime;
				std::ios_base::iostate err = std::ios_base::goodbit;
				tg.get(is, 0, is, err, &baseTime, std::begin(patternYMDHMS), std::end(patternYMDHMS) - 1);

				if(err != std::ios_base::goodbit) {
					return false;
				}

				base =
					CountDays(baseTime.tm_year + 1900, baseTime.tm_mon + 1, baseTime.tm_mday) * 24 * 60 * 60 +
					baseTime.tm_hour * 60 * 60 +
					baseTime.tm_min * 60 +
					baseTime.tm_sec * 1;
			}

			// Check if there is a time offset from UTC specified
			char sign = '\0';
			if(!is.eof()) {
				is.get(sign);
				if(is.rdstate() != std::ios_base::goodbit) {
					return false;
				}

				// Throw out fractional seconds
				if(sign == '.') {
					std::uint32_t frac = 0;
					is >> frac;

					// Try to get the sign again
					if(!is.eof()) {
						is.get(sign);
						if(is.rdstate() != std::ios_base::goodbit) {
							return false;
						}
					}
				}
			}

			// Handle specific time shift from UTC 
			if(sign == '+' || sign == '-') {
				std::tm offsetTime;
				char const patternHM[] = "%H:%M";

				std::ios_base::iostate err = std::ios_base::goodbit;
				tg.get(is, 0, is, err, &offsetTime, std::begin(patternHM), std::end(patternHM) - 1);

				if(!(err & std::ios_base::failbit)) {
					offset = (sign == '+' ? 1 : -1) * ((offsetTime.tm_hour * 3600) + (offsetTime.tm_min * 60));
				}
			}
			// UTC
			else if(sign == 'Z') {
				offset = 0;
			}

			result = system_clock::from_time_t(base - offset);
		} catch(...) {
			// Eat any errors
			return false;
		}

		return true;
	}

	inline void PrintByte(tostringstream& stream, std::uint8_t byte) {
		stream << std::setw(2) << std::setfill(_T('0')) << +byte;
	}
}


bool Twitch::IsValidChannelName(tstring const& channelName) {
	return IsValidUserName(channelName);
}


bool Twitch::IsValidUserName(tstring const& userName) {
	if(userName.empty()) {
		return false;
	}

	for(TCHAR character : userName) {
		if(!_istalnum(character) && (character != _T('_'))) {
			return false;
		}
	}
	return true;
}


bool Twitch::IsValidOAuthToken(tstring const& oauthToken) {
	return oauthToken != _T("");
}


int Twitch::IsWhitespace(int ch) {
	return ch == _T(' ') || ch == _T('\t') || ch == _T('\r') || ch == _T('\n');
}


tstring Twitch::ToLowerCase(tstring const& str) {
	tstring lower = str;
	std::transform(lower.begin(), lower.end(), lower.begin(), [](TCHAR ch) {
		return static_cast<TCHAR>(::_totlower(ch));
	});
	return lower;
}


void Twitch::Trim(tstring& str) {
	// trim leading and trailing whitespace
	str.erase(str.begin(), std::find_if(str.begin(), str.end(), [](TCHAR ch) { return !IsWhitespace(ch); }));
	str.erase(std::find_if(str.rbegin(), str.rend(), [](TCHAR ch) { return !IsWhitespace(ch); }).base(), str.end());
}


void Twitch::Split(tstring const& str, std::vector<tstring>& result, TCHAR sep, bool includeEmpty) {
	unsigned start = 0;
	unsigned end = 0;

	for(;;) {
		if(end == str.size() || str[end] == sep) {
			if(end > start || includeEmpty) {
				result.push_back(str.substr(start, end - start));
			}

			if(end == str.size()) {
				break;
			}

			++end;
			start = end;
		} else {
			++end;
		}
	}
}


void Twitch::Split(tstring const& str, tstring const& delim, std::vector<tstring>& result) {
	if(delim == _T("")) {
		result.push_back(str);
		return;
	}

	tstring scratch = str;

	while(!scratch.empty()) {
		size_t const index = scratch.find(delim);

		// Found a delimeter
		if(index != tstring::npos) {
			// Don't push empty tokens
			if(index > 0) {
				result.push_back(scratch.substr(0, index));
			}

			// Remove the token and the trailing delimeter
			scratch.erase(0, index + delim.size());
		}
		// No delimeters, the rest of the string is a token 
		else {
			result.push_back(scratch);

			break;
		}
	}
}


bool Twitch::StartsWith(tstring const& str, tstring const& prefix) {
	if(str.size() < prefix.size()) {
		return false;
	}

	return str.substr(0, prefix.size()) == prefix;
}


bool Twitch::EndsWith(tstring const& str, tstring const& suffix) {
	if(str.size() < suffix.size()) {
		return false;
	}

	tstring const tail = str.substr(str.size() - suffix.size(), suffix.size());

	return tail == suffix;
}


bool Twitch::RFC3339TimeToUnixTimestamp(const std::string& str, Timestamp& result) {
	result = 0;

	std::chrono::system_clock::time_point timePoint;
	if(!RFC3339TimeToTimePoint(str, timePoint)) {
		return false;
	}

	result = static_cast<Timestamp>(std::chrono::system_clock::to_time_t(timePoint));

	return true;
}


/**
* A better string copying function which guarantees null termination of the destination and
* doesn't pad the remainder of dst \0 if src is shorter than maxLen.
*/
void Twitch::SafeStringCopy(LPTSTR dst, string_t src, size_t maxLen) {
	for(size_t i = 0; i < maxLen; ++i) {
		dst[i] = src[i];

		if(src[i] == _T('\0')) {
			return;
		}
	}

	dst[maxLen - 1] = _T('\0');
}


tstring Twitch::GetGuid() {
	std::random_device device;
	std::default_random_engine engine(device());
	std::uniform_int_distribution<std::uint64_t> distribution(std::numeric_limits<std::uint64_t>::min(), std::numeric_limits<std::uint64_t>::max());

	std::uint64_t generated[2];
	generated[0] = distribution(engine);
	generated[1] = distribution(engine);

	std::uint8_t const* bytes = reinterpret_cast<std::uint8_t*>(&generated);

	tostringstream uuidStream;
	uuidStream << std::hex;
	PrintByte(uuidStream, bytes[0]);
	PrintByte(uuidStream, bytes[1]);
	PrintByte(uuidStream, bytes[2]);
	PrintByte(uuidStream, bytes[3]);
	uuidStream << _T("-");
	PrintByte(uuidStream, bytes[4]);
	PrintByte(uuidStream, bytes[5]);
	uuidStream << _T("-");
	PrintByte(uuidStream, bytes[6]);
	PrintByte(uuidStream, bytes[7]);
	uuidStream << _T("-");
	PrintByte(uuidStream, bytes[8]);
	PrintByte(uuidStream, bytes[9]);
	uuidStream << _T("-");
	PrintByte(uuidStream, bytes[10]);
	PrintByte(uuidStream, bytes[11]);
	PrintByte(uuidStream, bytes[12]);
	PrintByte(uuidStream, bytes[13]);
	PrintByte(uuidStream, bytes[14]);
	PrintByte(uuidStream, bytes[15]);

	return uuidStream.str();
}


/**
* Parse arguments for command-line samples. Essentially, splits on spaces while treating
* unescaped quoted segments as complete tokens.
*/
std::vector<tstring> Twitch::ParseArguments(tstring args) {
	std::vector<tstring> out;
	unsigned int currentIndex = 0;

	for(;;) {
		// Advance to beginning of next token.
		while(currentIndex < args.length() && Twitch::IsWhitespace(args[currentIndex])) {
			currentIndex++;
		}

		if(currentIndex >= args.length()) {
			break;
		}

		bool const tokenStartsWithQuote = args[currentIndex] == _T('"');

		if(tokenStartsWithQuote) {
			// If we start with a quote, we want to exclude it from the output string.
			currentIndex++;
		}

		unsigned int const tokenStart = currentIndex;

		for(;;) {
			if((Twitch::IsWhitespace(args[currentIndex]) && !tokenStartsWithQuote) || currentIndex >= args.length()) {
				out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
				break;
			}

			bool currentIsUnescapedQuote = args[currentIndex] == _T('"') && currentIndex > 0 && args[currentIndex - 1] != _T('\\');
			if(currentIsUnescapedQuote && tokenStartsWithQuote) {
				// End of token formed by quotes
				out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
				break;
			} else if(currentIsUnescapedQuote && !tokenStartsWithQuote) {
				// Token ended by quote in middle of string. Turn back time by one so we treat the
				// quote as the started for the next token.
				out.push_back(args.substr(tokenStart, currentIndex - tokenStart));
				currentIndex--;
				break;
			}

			currentIndex++;
		}

		currentIndex++;
	}

	return out;
}
