// Copyright (c) 2019-2021, Twitch Interactive, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
// OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// OTHER DEALINGS IN THE SOFTWARE.

#pragma once

#include <cassert>
#include <cstddef>
#include <cstdint>
#include <ctime>
#include <condition_variable>
#include <exception>
#include <functional>
#include <future>
#include <map>
#include <memory>
#include <mutex>
#include <ostream>
#include <string>
#include <type_traits>
#include <utility>
#include <vector>
#ifdef __NX__
# include <nn/nn_TimeSpan.h>
#endif

namespace Twitch {
#ifdef _UNICODE
	using string_t = wchar_t const*;
	using tstring = std::wstring;
#else
	using string_t = char const*;
	using tstring = std::string;
#endif
	using stringmap = std::map<tstring, tstring>;

	class TwitchException;
	template<typename T> class Pollable;
	class Callback;

#ifdef __NX__
	bool Initialize(nn::TimeSpan timeout = nn::TimeSpan::FromSeconds(10));
#endif
#ifdef _UNICODE
	std::string FromTstring(tstring const& s);
	tstring ToTstring(std::string const& s);
	template<typename T>
	inline tstring to_tstring(T value) { return std::to_wstring(value); }
#else
	inline std::string FromTstring(tstring const& s) { return s; }
	inline tstring ToTstring(std::string const& s) { return s; }
	template<typename T>
	inline tstring to_tstring(T value) { return std::to_string(value); }
#endif

	template <typename...>
	using void_t = void;

	template <typename, typename = void>
	struct has_mapped_type : std::false_type {};

	template <typename T>
	struct has_mapped_type<T, void_t<typename T::mapped_type>> : std::true_type {};

	template <typename, typename = void>
	struct has_const_iterator : std::false_type {};

	template <typename T>
	struct has_const_iterator<T, void_t<typename T::const_iterator>> : std::true_type {};

	template<typename T>
	using is_character = std::integral_constant<bool, std::is_same<char, T>::value || std::is_same<char16_t, T>::value ||
		std::is_same<char32_t, T>::value || std::is_same<wchar_t, T>::value>;

	template<typename T>
	using is_string = std::integral_constant<bool, std::is_same<tstring, T>::value>;

	template<typename T>
	using is_integer = std::integral_constant<bool, std::is_integral<T>::value && !std::is_same<bool, T>::value && !is_character<T>::value>;

	template<typename T>
	using is_iterable = std::integral_constant<bool, has_const_iterator<T>::value && !has_mapped_type<T>::value && !is_string<T>::value>;

	template<typename T>
	using is_string_mapped = std::integral_constant<bool, has_mapped_type<T>::value && is_string<typename T::key_type>::value>;

	template<typename T>
	using is_integer_mapped = std::integral_constant<bool, has_mapped_type<T>::value && is_integer<typename T::key_type>::value>;

	void Stringify(std::ostream& os, double d);
	void Stringify(std::ostream& os, char const* s);
#ifdef _UNICODE
	void Stringify(std::ostream& os, wchar_t const* s);
#endif

	inline void Stringify(std::ostream& os, nullptr_t) {
		os << "null";
	}

	inline void Stringify(std::ostream& os, bool b) {
		auto const* const s = b ? "true" : "false";
		os << s;
	}

	inline void Stringify(std::ostream& os, std::string const& s) {
		Stringify(os, s.c_str());
	}

#ifdef _UNICODE
	inline void Stringify(std::ostream& os, std::wstring const& s) {
		Stringify(os, s.c_str());
	}
#endif

	inline void Stringify(std::ostream& os, char ch) {
		char s[2]{ ch };
		Stringify(os, s);
	}

#ifdef _UNICODE
	inline void Stringify(std::ostream& os, wchar_t ch) {
		wchar_t s[2]{ ch };
		Stringify(os, s);
	}
#endif

	template<typename T>
	std::enable_if_t<is_integer<T>::value> Stringify(std::ostream& os, T const& t) {
		os << t;
	}

	template<>
	inline void Stringify(std::ostream& os, std::uint8_t const& t) {
		os << static_cast<unsigned>(t);
	}

	template<typename T>
	std::enable_if_t<is_string_mapped<T>::value> Stringify(std::ostream& os, T const& m) {
		os << '{';
		for(typename T::const_iterator it = m.cbegin(), end = m.cend(); it != end; ) {
			StringifyObjectMember(os, it->first, it->second);
			if(++it != end) {
				os << ',';
			}
		}
		os << '}';
	}

	template<typename T>
	std::enable_if_t<is_integer_mapped<T>::value> Stringify(std::ostream& os, T const& m) {
		os << '{';
		for(typename T::const_iterator it = m.cbegin(), end = m.cend(); it != end; ) {
			StringifyObjectMember(os, to_tstring(it->first), it->second);
			if(++it != end) {
				os << ',';
			}
		}
		os << '}';
	}

	template<typename T>
	std::enable_if_t<is_iterable<T>::value> Stringify(std::ostream& os, T const& m) {
		os << '[';
		for(typename T::const_iterator it = m.cbegin(), end = m.cend(); it != end; ) {
			Stringify(os, *it);
			if(++it != end) {
				os << ',';
			}
		}
		os << ']';
	}

	template<typename T, size_t N>
	void Stringify(std::ostream& os, T const (&m)[N]) {
		os << '[';
		for(size_t i = 0; i < N; ++i) {
			Stringify(os, m[i]);
			if(i < N - 1) {
				os << ',';
			}
		}
		os << ']';
	}

	template<typename T>
	void StringifyObjectMember(std::ostream& os, tstring const& name, T const& value) {
		Stringify(os, name);
		os << ':';
		Stringify(os, value);
	}

	template<typename T, typename... Types>
	void StringifyObjectMember(std::ostream& os, tstring const& name, T const& value, Types... rest) {
		StringifyObjectMember(os, name, value);
		os << ',';
		StringifyObjectMember(os, rest...);
	}

	template<typename... Types>
	void StringifyObject(std::ostream& os, Types... rest) {
		os << '{';
		StringifyObjectMember(os, rest...);
		os << '}';
	}
}

class Twitch::TwitchException : public std::exception {
public:
	explicit TwitchException(int errorCode) noexcept;

	char const* what() const noexcept override;
	int GetErrorCode() const noexcept { return errorCode; }

	__declspec(property(get = GetErrorCode)) int const ErrorCode;

private:
	char description[128]{};
	int errorCode;
};

template<typename T>
class Twitch::Pollable {
public:
	enum class PollableReadyState { NotReady, ValueReady, ExceptionReady };

	explicit Pollable(std::future<T>&& deferredValue) : value(), readyState(PollableReadyState::NotReady) {
		auto fn = [this](std::future<T>&& deferredValue) {
			try {
				value = deferredValue.get();
				readyState = PollableReadyState::ValueReady;
			} catch(...) {
				ex = std::current_exception();
				readyState = PollableReadyState::ExceptionReady;
			}
		};
		task = std::async(std::launch::async, fn, std::move(deferredValue));
	}

	PollableReadyState GetReadyState() const noexcept { return readyState; }
	bool GetIsReady() const noexcept { return readyState != PollableReadyState::NotReady; }
	bool GetHasValue() const noexcept { return readyState == PollableReadyState::ValueReady; }
	bool GetHasException() const noexcept { return readyState == PollableReadyState::ExceptionReady; }
	T const& GetValue() const {
		if(GetHasException()) {
			std::rethrow_exception(ex);
		} else {
			return value;
		}
	}
	std::exception_ptr GetException() const noexcept { return ex; }

	__declspec(property(get = GetReadyState)) PollableReadyState const ReadyState;
	__declspec(property(get = GetIsReady)) bool const IsReady;
	__declspec(property(get = GetHasValue)) bool const HasValue;
	__declspec(property(get = GetHasException)) bool const HasException;
	__declspec(property(get = GetValue)) T const& Value;
	__declspec(property(get = GetException)) std::exception_ptr const Exception;

private:
	std::future<void> task;
	T value;
	std::exception_ptr ex;
	PollableReadyState readyState;
};

class Twitch::Callback {
public:
	Callback() = default;
	Callback(Callback const&) = delete;
	Callback(Callback&&) = default;
	Callback& operator=(Callback const&) = delete;
	Callback& operator=(Callback&&) = default;
	template<typename T, typename S, typename F>
	Callback(std::future<T>&& deferredValue, S onSuccess, F onFailure) {
		auto fn = [onSuccess, onFailure](std::future<T>&& deferredValue) {
			T value;
			try {
				value = deferredValue.get();
			} catch(...) {
				try {
					onFailure(std::current_exception());
				} catch(...) {
					// The onFailure call-back threw an exception.  These are
					// silently ignored in release builds.
					assert(false);
				}
				return;
			}
			try {
				onSuccess(value);
			} catch(...) {
				// The onSuccess call-back threw an exception.  These are
				// silently ignored in release builds.
				assert(false);
			}
		};
		task = std::async(std::launch::async, fn, std::move(deferredValue));
	}
	~Callback() = default;

	void CheckForException() {
		if(task.valid()) {
			task.get();
		}
	}

private:
	std::future<void> task;
};
