#include "pch.h"
#include "../Shared/Internal.h"
#include "../Shared/HttpUtil.h"
#include "../Shared/Json.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

template<typename T>
std::string TestStringify(T const& t) {
	std::stringstream ss;
	Stringify(ss, t);
	return ss.str();
}

namespace {
	class JsonTestClass {
	public:
		explicit JsonTestClass(unsigned u) : u(u), a{ 1, 2, 3, 4, 5, 6, 7, 8 } {}

	private:
		friend void Stringify(std::ostream& os, JsonTestClass const& t);

		unsigned u;
		std::array<std::uint8_t, 9> a;
	};

	void Stringify(std::ostream& os, JsonTestClass const& t) {
		StringifyObject(os, _T("u"), t.u, _T("a"), t.a);
	}
}

namespace WindowsLibTest {
	TEST_CLASS(JsonTest) {
public:
	TEST_METHOD(JsonBoolean) {
		auto const value = Json(true);
		auto const expectedValue = "true\n"s;
		auto const actualValue = value.dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonCharacterPointer) {
		char const* const value = "string";
		auto const expectedValue = '"' + std::string(value) + "\"\n";
		auto const actualValue = Json(value).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonInteger) {
		int const value = 9;
		auto const expectedValue = std::to_string(value) + '\n';
		auto const actualValue = Json(value).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonShortArray) {
		short const value[] = { 1, 2, 3 };
		Json::array array;
		for (auto& i : value) {
			array.push_back(i);
		}
		auto const expectedValue = "[1,2,3]\n"s;
		auto const actualValue = Json(array).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonString) {
		std::string const value = "string"s;
		auto const expectedValue = '"' + value + "\"\n";
		auto const actualValue = Json(value).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonStringArray) {
		std::string const value[] = { "one"s, "two"s, "three"s };
		Json::array array;
		for (auto& s : value) {
			array.push_back(s);
		}
		auto const expectedValue = "[\"one\",\"two\",\"three\"]\n"s;
		auto const actualValue = Json(array).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(JsonObject) {
		Json::object object;
		object["zero"] = false;
		object["one"] = 1;
		object["two"] = "two";
		auto const expectedValue = "{\"one\":1,\"two\":\"two\",\"zero\":false}\n"s;
		auto const actualValue = Json(object).dump();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(IsArraySuccess) {
		short const value[] = { 1, 2, 3 };
		Json::array array;
		for (auto& i : value) {
			array.push_back(i);
		}
		auto const actualValue = Json(array).is_array();
		Assert::IsTrue(actualValue);
	}

	TEST_METHOD(IsArrayFailure) {
		auto const actualValue = Json("string").is_array();
		Assert::IsFalse(actualValue);
	}

	TEST_METHOD(IsNumberSuccess) {
		auto const actualValue = Json(1).is_number();
		Assert::IsTrue(actualValue);
	}

	TEST_METHOD(IsNumberFailure) {
		auto const actualValue = Json("string").is_number();
		Assert::IsFalse(actualValue);
	}

	TEST_METHOD(IsObjectSuccess) {
		Json::object object;
		object["zero"] = false;
		object["one"] = 1;
		object["two"] = "two";
		auto const actualValue = Json(object).is_object();
		Assert::IsTrue(actualValue);
	}

	TEST_METHOD(IsObjectFailure) {
		auto const actualValue = Json("string").is_object();
		Assert::IsFalse(actualValue);
	}

	TEST_METHOD(IsStringSuccess) {
		auto const actualValue = Json("string").is_string();
		Assert::IsTrue(actualValue);
	}

	TEST_METHOD(IsStringFailure) {
		auto const actualValue = Json(1).is_string();
		Assert::IsFalse(actualValue);
	}

	TEST_METHOD(BoolValueSuccess) {
		auto const actualValue = Json(true).bool_value();
		Assert::IsTrue(actualValue);
	}

	TEST_METHOD(BoolValueFailure) {
		auto const actualValue = Json("string").bool_value();
		Assert::IsFalse(actualValue);
	}

	TEST_METHOD(StringValueSuccess) {
		auto const expectedValue = "string"s;
		auto const actualValue = Json(expectedValue).string_value();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringValueFailure) {
		auto const expectedValue = ""s;
		auto const actualValue = Json(true).string_value();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(IntValueSuccess) {
		auto const expectedValue = 1;
		auto const actualValue = Json(1).int_value();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(IntValueFailure) {
		auto const expectedValue = 0;
		auto const actualValue = Json("string").int_value();
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(ParseSuccess) {
		auto const value = "{\"u\":9,\"a\":[1,2,3,4,5,6,7,8,0]}"s;
		std::string error;
		auto const actualValue = Json::Parse(value, error);
		Assert::IsTrue(error.empty());
		Assert::IsTrue(actualValue.is_object());
		Assert::AreEqual(size_t(2), actualValue.object_items().size());
	}

	TEST_METHOD(ParseFailure) {
		auto const value = "{\"u\":9,\"a\":["s;
		std::string error;
		auto const actualValue = Json::Parse(value, error);
		Assert::IsFalse(error.empty());
		Assert::AreEqual("null\n"s, actualValue.dump());
	}

	TEST_METHOD(StringifyBoolean) {
		bool const value = true;
		auto const expectedValue = "true"s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyCharacter) {
		char const value = 'c';
		auto const expectedValue = '"' + std::string(1, value) + '"';
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyCharacterPointer) {
		char const* const value = "string";
		auto const expectedValue = '"' + std::string(value) + '"';
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyClass) {
		auto const value = JsonTestClass(9u);
		auto const expectedValue = "{\"u\":9,\"a\":[1,2,3,4,5,6,7,8,0]}"s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyDouble) {
		double const value = 9. + 1. / 64;
		auto expectedValue = std::to_string(value);
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyFloat) {
		float const value = 9.f + 1.f / 64;
		auto const expectedValue = std::to_string(value);
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyInteger) {
		int const value = 9;
		auto const expectedValue = std::to_string(value);
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyNull) {
		nullptr_t const value = nullptr;
		auto const expectedValue = "null"s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyShortArray) {
		short const value[] = { 1, 2, 3 };
		auto const expectedValue = "[1,2,3]"s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyString) {
		std::string const value = "string"s;
		auto const expectedValue = '"' + value + '"';
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyStringArray) {
		std::string const value[] = { "one"s, "two"s, "three"s };
		auto const expectedValue = "[\"one\",\"two\",\"three\"]"s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
	}

	TEST_METHOD(StringifyWideCharacterPointer) {
#ifdef _UNICODE
		wchar_t const* const value = L"string";
		auto const expectedValue = '"' + FromTstring(value) + '"';
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
#endif
	}

	TEST_METHOD(StringifyWideCharacter) {
#ifdef _UNICODE
		wchar_t const value = L'\x0102';
		auto const expectedValue = "\"\xc4\x82\""s;
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
#endif
	}

	TEST_METHOD(StringifyWideString) {
#ifdef _UNICODE
		std::wstring const value = L"string"s;
		auto const expectedValue = '"' + FromTstring(value) + '"';
		auto const actualValue = TestStringify(value);
		Assert::AreEqual(expectedValue, actualValue);
#endif
	}
	};
}
