#include "pch.h"
#include "../Shared/Internal.h"
#include "../Shared/HttpUtil.h"

using namespace std::literals;
using namespace Microsoft::VisualStudio::CppUnitTestFramework;
using namespace Twitch;

#if defined(_WIN32) && !defined(_XBOX_ONE)
template<>
std::wstring Microsoft::VisualStudio::CppUnitTestFramework::ToString<std::future_status>(std::future_status const& status) {
	switch(status) {
	case std::future_status::ready: return L"ready";
	case std::future_status::timeout: return L"timeout";
	case std::future_status::deferred: return L"deferred";
	default: return L"(unknown)";
	}
}
#endif

namespace WindowsLibTest {
	TEST_CLASS(UtilitiesTest) {
private:
	static auto CreateCallbackTest(bool wantsFailure, std::promise<int>& successPromise, std::promise<std::string>& errorPromise) {
		auto onSuccess = [&successPromise](int value) {
			successPromise.set_value(value);
		};
		auto onError = [&errorPromise](std::exception_ptr pex) {
			try {
				std::rethrow_exception(pex);
			} catch(std::exception const& ex) {
				errorPromise.set_value(ex.what());
			}
		};
#ifdef _WIN32
		return Callback(CreateDeferredValue(wantsFailure), onSuccess, onError);
#else
		UNREFERENCED_PARAMETER(wantsFailure);
		UNREFERENCED_PARAMETER(onSuccess);
		UNREFERENCED_PARAMETER(onError);
		return nullptr;
#endif
	}

public:
	TEST_METHOD(Random_Success) {
#ifndef _XBOX_ONE
		unsigned value;
		rand_s(&value);
#endif
	}

	TEST_METHOD(UrlEncode_Success) {
#ifdef _UNICODE
		auto expected = tstring(_T("%E0%A6%9A!b@c%23d%24e%25f%5Eg%26h*i(j)k-l_m%3Dn%2Bo%60p~q%5Br%7Bs%5Dt%7Du%5Cv%7Cw;x:y'z%220%2F1%3F2,3%3C4.5%3E%C2%99"));
		auto actual = UrlEncode(_T("\x99a!b@c#d$e%f^g&h*i(j)k-l_m=n+o`p~q[r{s]t}u\\v|w;x:y'z\"0/1?2,3<4.5>\x99"));
#else
		auto expected = tstring(_T("%99%20a!b@c%23d%24e%25f%5Eg%26h*i(j)k-l_m%3Dn%2Bo%60p~q%5Br%7Bs%5Dt%7Du%5Cv%7Cw;x:y'z%220%2F1%3F2,3%3C4.5%3E%99"));
		auto actual = UrlEncode(_T("\x99 a!b@c#d$e%f^g&h*i(j)k-l_m=n+o`p~q[r{s]t}u\\v|w;x:y'z\"0/1?2,3<4.5>\x99"));
#endif
		Assert::AreEqual(expected, actual);
	}

	TEST_METHOD(Callback_Success) {
		std::promise<int> successPromise;
		std::promise<std::string> errorPromise;
		auto callback = CreateCallbackTest(false, successPromise, errorPromise);
		UNREFERENCED_PARAMETER(callback);
#ifdef _WIN32
		Assert::AreEqual(successValue, successPromise.get_future().get());
#endif
		Assert::AreEqual(std::future_status::timeout, errorPromise.get_future().wait_for(0s));
	}

	TEST_METHOD(Callback_Failure) {
		std::promise<int> successPromise;
		std::promise<std::string> errorPromise;
		auto callback = CreateCallbackTest(true, successPromise, errorPromise);
		UNREFERENCED_PARAMETER(callback);
#ifdef _WIN32
		Assert::AreEqual(failureValue, errorPromise.get_future().get().c_str());
#endif
		Assert::AreEqual(std::future_status::timeout, successPromise.get_future().wait_for(0s));
	}

	TEST_METHOD(QueryableFuture_Success) {
#ifdef _WIN32
		Pollable<int> pollable(CreateDeferredValue(false));
		std::this_thread::sleep_for(11ms);
		Assert::IsTrue(pollable.IsReady);
		Assert::AreEqual(successValue, pollable.Value);
#endif
	}

	TEST_METHOD(QueryableFuture_Failure) {
#ifdef _WIN32
		Pollable<int> pollable(CreateDeferredValue(true));
		std::this_thread::sleep_for(11ms);
		Assert::IsTrue(pollable.IsReady);
		std::string actualValue;
		try {
			std::rethrow_exception(pollable.Exception);
		} catch(std::exception const& ex) {
			actualValue = ex.what();
		}
		Assert::AreEqual(failureValue, actualValue.c_str());
#endif
	}

	TEST_METHOD(FromTstring_Success) {
		auto s = FromTstring(_T("test"));
		Assert::IsFalse(s.empty());
	}

	TEST_METHOD(FromTstring_Failure) {
#ifdef _UNICODE
		Assert::ExpectException<TwitchException>([] {
			FromTstring(_T("\xdddd"));
		});
#endif
	}

	TEST_METHOD(ToTstring_Success) {
		auto s = ToTstring("test");
		Assert::IsFalse(s.empty());
	}

	TEST_METHOD(ToTstring_Failure) {
#ifdef _UNICODE
		Assert::ExpectException<TwitchException>([] {
			ToTstring("\x80");
		});
#endif
	}

	TEST_METHOD(to_tstring_Success) {
		auto s = to_tstring(9);
		Assert::AreEqual(_T("9"), s.c_str());
	}

private:
	static constexpr char failureValue[] = "failure";
	static constexpr int successValue = 22;

	static std::future<int> CreateDeferredValue(bool wantsFailure) {
		return std::async(std::launch::deferred, [wantsFailure] {
			if(wantsFailure) {
				throw std::runtime_error(failureValue);
			}
			return successValue;
		});
	}
	};

#ifdef __ORBIS__
	constexpr char UtilitiesTest::failureValue[];
#endif
}
