package buffered_logger

import (
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/sirupsen/logrus"

	logging "code.justin.tv/amzn/TwitchLogging"
	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
)

const (
	defaultReportPeriod = 1 * time.Second
	droppedLogsMetrics  = "NumOfDroppedLogs"
	droppedLogsKey      = "DroppedLogs"
	timestampKey        = "Timestamp"
	maxLogEventSize     = 10000
)

type LogIdentifier struct {
	// Used to filter if this hook should be triggered with current log event
	Name   string
	Group  string
	Stream string
}

type Batch struct {
	events []*cloudwatchlogs.InputLogEvent
	size   int
}

// Since PutLogEvents has rate limit 5 requests per second
// So this hook buffered logs sent with logrus and periodically report to CW logs
type CWLogHook struct {
	client        *cloudwatchlogs.CloudWatchLogs
	triggerLevels []logrus.Level
	identifier    LogIdentifier
	// A buffer to hold all the events to be processed
	buffer            chan cloudwatchlogs.InputLogEvent
	nextSequenceToken *string
	lock              sync.Mutex
	bufferPeriod      time.Duration
	closeSignal       chan bool
	batch             Batch
	// the logger to handle dropped events and other unexpected failures
	// usually it could be s simple log to disk or to your own error monitoring (e.g rollbar)
	logger logging.Logger
	pid    identifier.ProcessIdentifier
	// Used to report dropped metrics
	reporter *telemetry.SampleReporter
}

func NewBufferedCWHookWithDefaultSettings(sess *session.Session, pid identifier.ProcessIdentifier, logIdentifier LogIdentifier) (*CWLogHook, error) {
	defaultLevels := []logrus.Level{
		logrus.PanicLevel,
		logrus.FatalLevel,
		logrus.ErrorLevel,
		logrus.WarnLevel,
		logrus.InfoLevel,
		logrus.DebugLevel,
	}

	hook := &CWLogHook{
		client:            cloudwatchlogs.New(sess),
		triggerLevels:     defaultLevels,
		nextSequenceToken: nil,
		lock:              sync.Mutex{},
		buffer:            make(chan cloudwatchlogs.InputLogEvent, maxLogEventSize),
		closeSignal:       make(chan bool, 1),
		pid:               pid,
		bufferPeriod:      defaultReportPeriod,
		identifier:        logIdentifier,
		batch: Batch{
			events: nil,
			size:   0,
		},
	}

	if err := hook.getCWLogGroup(); err != nil {
		return hook, err
	}

	hook.runWorkToBeDone()

	return hook, nil
}

func NewBufferedCWHook(sess *session.Session, reporter *telemetry.SampleReporter, triggerLevels []logrus.Level, logger logging.Logger, pid identifier.ProcessIdentifier, logIdentifier LogIdentifier, bufferPeriod time.Duration) (*CWLogHook, error) {
	hook := &CWLogHook{
		client:            cloudwatchlogs.New(sess),
		triggerLevels:     triggerLevels,
		nextSequenceToken: nil,
		lock:              sync.Mutex{},
		buffer:            make(chan cloudwatchlogs.InputLogEvent, maxLogEventSize),
		closeSignal:       make(chan bool, 1),
		logger:            logger,
		pid:               pid,
		bufferPeriod:      bufferPeriod,
		reporter:          reporter,
		identifier:        logIdentifier,
		batch: Batch{
			events: nil,
			size:   0,
		},
	}

	if err := hook.getCWLogGroup(); err != nil {
		return hook, err
	}

	hook.runWorkToBeDone()

	return hook, nil
}

func (hook *CWLogHook) Levels() []logrus.Level {
	return hook.triggerLevels
}
