package buffered_logger

import (
	"fmt"
	"os"
	"testing"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/stretchr/testify/assert"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/sirupsen/logrus"
)

func TestHook(t *testing.T) {
	key := os.Getenv("AWS_ACCESS_KEY")
	secret := os.Getenv("AWS_SECRET_KEY")
	group := os.Getenv("AWS_CLOUDWATCHLOGS_GROUP_NAME")
	stream := os.Getenv("AWS_CLOUDWATCHLOGS_STREAM_NAME1")
	stream2 := os.Getenv("AWS_CLOUDWATCHLOGS_STREAM_NAME2")

	if key == "" {
		t.Skip("skipping test; AWS_ACCESS_KEY not set")
	}

	if secret == "" {
		t.Skip("skipping test; AWS_SECRET_KEY not set")
	}

	if group == "" {
		t.Skip("skipping test; AWS_CLOUDWATCHLOGS_GROUP_NAME not set")
	}

	if stream == "" {
		t.Skip("skipping test; AWS_CLOUDWATCHLOGS_STREAM_NAME not set")
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String("us-west-2"),
	})

	assert.NoError(t, err)

	hook, err := NewBufferedCWHookWithDefaultSettings(sess, identifier.ProcessIdentifier{
		Service: "TestService",
		Region:  "us-west-2",
		Stage:   "Development",
	}, LogIdentifier{
		Name:   "TestBusinessMetrics",
		Group:  group,
		Stream: stream,
	})

	assert.NoError(t, err)
	assert.NotNil(t, hook)

	ticker := time.NewTicker(500 * time.Millisecond)
	done := make(chan bool)

	hook2, err := NewBufferedCWHookWithDefaultSettings(sess, identifier.ProcessIdentifier{
		Service: "TestService2",
		Region:  "us-west-2",
		Stage:   "Development2",
	}, LogIdentifier{
		Name:   "TestBusinessMetrics2",
		Group:  group,
		Stream: stream2,
	})

	log1 := logrus.New()
	log1.AddHook(hook)
	log1.AddHook(hook2)

	go func() {
		count := 1
		for {
			select {
			case <-done:
				hook.Stop()
				// hook2.Stop()
				return
			case <-ticker.C:
				if count > 4 {
					continue
				}
				// Should show up in CW logs
				log1.WithFields(logrus.Fields{
					CwLogName:    "TestBusinessMetrics",
					"TwitchID":   "123",
					"Count":      count,
					"TwitchName": "LogTest",
					"Action":     "download_games_report",
				}).Info("Good Info Test")

				// Should show up in CW logs
				log1.WithFields(logrus.Fields{
					CwLogName:    "TestBusinessMetrics",
					"TwitchID":   "999",
					"Count":      count,
					"TwitchName": "LogTest",
					"Action":     "retrieve_games_report",
				}).Info("Good Info Test Follow")

				// Should show not up in CW logs
				log1.WithFields(logrus.Fields{
					CwLogName:    "BadTestBusinessMetrics",
					"TwitchID":   "456",
					"Count":      count,
					"TwitchName": "BadLogTest",
					"Action":     "bad_download_games_report",
				}).Info("Bad Info Test")

				// Should show up in CW logs2
				log1.WithFields(logrus.Fields{
					CwLogName:    "TestBusinessMetrics2",
					"TwitchID":   "123",
					"Count":      count,
					"TwitchName": "LogTest2",
					"Action":     "upload_games_report",
				}).Info("Second Good Info Test")
			}

			count++
		}
	}()

	time.Sleep(3000 * time.Millisecond)
	ticker.Stop()
	done <- true
	fmt.Println("Ticker stopped")

	time.Sleep(2000 * time.Millisecond)
}
