import cdk = require('@aws-cdk/core');
import { ConfigServiceStack } from './stacks/config-service';
import { ConfigServiceReposStack } from './stacks/config-service-repo';
import { ConfigServiceSecretsStack } from './stacks/config-service-secrets';
import { EventBusStack } from './stacks/event-bus';
import { InterfaceVpcEndpointStack } from './stacks/interface-vpc-endpoint';
import { JenkinsUserStack } from './stacks/jenkins-user';
import { PdmsLambdaStack } from './stacks/pdms-lambda';
import { S3BucketStack } from './stacks/s3-bucket';
import { SnsPagerdutyStack } from './stacks/sns-pagerduty';
import { VpcStack } from './stacks/vpc';
import { ZoneStack } from './stacks/zone-stack';

const env = {
  account: '925711164288',
  region: 'us-west-2',
};

const CONFIG_SERVICE_CATALOG_ID = '429';
const CONFIG_SERVICE_HOSTNAME = 'dev.configuration.extensions.twitch.a2z.com';
const OWL_HOSTNAME = 'prod.owl.twitch.a2z.com'; // use prod.owl because cartman-dev also uses prod.owl
const CARTMAN_HOSTNAME = 'cartman-dev.twitch.a2z.com';

export function createDevStacks(app: cdk.App) {
  const vpcStack = new VpcStack(app, 'DevVpc', {
    env,
    vpcId: 'vpc-051c0f591d3711c98',
  });

  const zoneStack = new ZoneStack(app, 'DevHostedZone', {
    env,
    zoneAttrs: {
      zoneName: CONFIG_SERVICE_HOSTNAME,
      hostedZoneId: 'Z1I3NVBLKIEY0C',
    },
  });

  const configServiceSecretsStack = new ConfigServiceSecretsStack(app, 'DevConfigServiceSecrets', { env });
  const configServiceReposStack = new ConfigServiceReposStack(app, 'DevConfigServiceRepo', { env });

  const s3BucketStack = new S3BucketStack(app, 'DevS3Bucket', { env });

  const snsPagerdutyStack = new SnsPagerdutyStack(app, 'DevPagerdutySns', {
    env,
    highUrgencyUrl: 'https://events.pagerduty.com/integration/0cea65cf29a949d4979759c36f8d9907/enqueue',
    lowUrgencyUrl: 'https://events.pagerduty.com/integration/c508ebff0d8741069caffae483c647a0/enqueue',
  });

  new InterfaceVpcEndpointStack(app, 'DevPrivateLinkOwl', {
    env,
    vpc: vpcStack.vpc,
    domainName: OWL_HOSTNAME,
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-08fdcf80669bbcb87', port: 443 },
  });

  new InterfaceVpcEndpointStack(app, 'DevPrivateLinkCartman', {
    env,
    vpc: vpcStack.vpc,
    domainName: CARTMAN_HOSTNAME,
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-0d48ba120d8a41fe2', port: 443 },
  });

  const eventBusCfn = new EventBusStack(app, 'DevEventBus', { env });

  const pdmsLambdaStack = new PdmsLambdaStack(app, 'DevPdmsLambda', {
    env,
    vpc: vpcStack.vpc,
    eventBusCfn,
    eventBusSqsQueueName: 'DevPdmsLambda-eventbus-queue',
    eventBusSqsDeadletterqueueName: 'DevPdmsLambda-eventbus-deadletterqueue',
    s3Bucket: s3BucketStack.bucket,
    s3CodeFile: 'lambda_go_build.zip', // lambda needs something to start, this can be pushed with make upload-code-to-s3 in https://git-aws.internal.justin.tv/devhub/config-service-pdms-lambda
    authClientId: 'ixcntszjbdj6m0afu4wizfmyxhs2yq',
    authClientSecret: configServiceSecretsStack.pdmsLambdaClientSecret,
    owlHost: new cdk.StringConcat().join('https://', OWL_HOSTNAME),
    cartmanHost: new cdk.StringConcat().join('https://', CARTMAN_HOSTNAME),
    configServiceHost: new cdk.StringConcat().join('http://', CONFIG_SERVICE_HOSTNAME),
    configServiceId: CONFIG_SERVICE_CATALOG_ID,
    pdmsApiLambdaArn: 'arn:aws:lambda:us-west-2:895799599216:function:PDMSLambda-LambdaFunction-IT8I1PE1YR81:live',
    pdmsAsumeRoleArn: 'arn:aws:iam::895799599216:role/PDMSLambda-CallerRole-18451FI19HSXT', // prod: arn:aws:iam::125704643346:role/PDMSLambda-CallerRole-13IIND444YKVR
    alarmTopic: snsPagerdutyStack.lowUrgencyTopic,
  });

  new ConfigServiceStack(app, 'DevConfigService', {
    datadogSecret: configServiceSecretsStack.datadog,
    env,
    networkLoadBalancerAttrs: {
      loadBalancerArn:
        'arn:aws:elasticloadbalancing:us-west-2:925711164288:loadbalancer/net/a6b0ddfdfcb4f11e8bb230a9ffd4b090/f8cc2bf0c78ab212',
      loadBalancerCanonicalHostedZoneId: 'Z18D5FSROUN65G',
      loadBalancerDnsName: 'a6b0ddfdfcb4f11e8bb230a9ffd4b090-f8cc2bf0c78ab212.elb.us-west-2.amazonaws.com',
    },
    repository: configServiceReposStack.primary,
    sandstormRoleArn:
      'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/extensions-configservice-dev',
    vpc: vpcStack.vpc,
    zone: zoneStack.zone,
  });

  new JenkinsUserStack(app, 'DevJenkinsUser', {
    env,
    lambdaFunctionArn: pdmsLambdaStack.lambdaFn.functionArn,
    s3BucketArn: s3BucketStack.bucket.bucketArn,
  });
}
