import cdk = require('@aws-cdk/core');
import { ConfigServiceStack } from './stacks/config-service';
import { ConfigServiceReposStack } from './stacks/config-service-repo';
import { ConfigServiceSecretsStack } from './stacks/config-service-secrets';
import { EventBusStack } from './stacks/event-bus';
import { InterfaceVpcEndpointStack } from './stacks/interface-vpc-endpoint';
import { JenkinsUserStack } from './stacks/jenkins-user';
import { PdmsLambdaStack } from './stacks/pdms-lambda';
import { S3BucketStack } from './stacks/s3-bucket';
import { SnsPagerdutyStack } from './stacks/sns-pagerduty';
import { VpcStack } from './stacks/vpc';
import { ZoneStack } from './stacks/zone-stack';

const env = {
  account: '816111448886',
  region: 'us-west-2',
};

const CONFIG_SERVICE_CATALOG_ID = '429';
const CONFIG_SERVICE_HOSTNAME = 'prod.configuration.extensions.twitch.a2z.com';
const OWL_HOSTNAME = 'prod.owl.twitch.a2z.com';
const CARTMAN_HOSTNAME = 'cartman.twitch.a2z.com';

export function createProdStacks(app: cdk.App) {
  const vpcStack = new VpcStack(app, 'ProdVpc', {
    env,
    vpcId: 'vpc-02a64f8b9559c479f',
  });

  const zoneStack = new ZoneStack(app, 'ProdHostedZone', {
    env,
    zoneAttrs: {
      zoneName: CONFIG_SERVICE_HOSTNAME,
      hostedZoneId: 'Z2Q2WEUX4ENBBP',
    },
  });

  const configServiceSecretsStack = new ConfigServiceSecretsStack(app, 'ProdConfigServiceSecrets', { env });
  const configServiceReposStack = new ConfigServiceReposStack(app, 'ProdConfigServiceRepos', { env });

  const s3BucketStack = new S3BucketStack(app, 'ProdS3Bucket', { env });

  const snsPagerdutyStack = new SnsPagerdutyStack(app, 'ProdPagerdutySns', {
    env,
    highUrgencyUrl: 'https://events.pagerduty.com/integration/0cea65cf29a949d4979759c36f8d9907/enqueue',
    lowUrgencyUrl: 'https://events.pagerduty.com/integration/c508ebff0d8741069caffae483c647a0/enqueue',
  });

  new InterfaceVpcEndpointStack(app, 'ProdPrivateLinkOwl', {
    env,
    vpc: vpcStack.vpc,
    domainName: OWL_HOSTNAME,
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-08fdcf80669bbcb87', port: 443 },
  });

  new InterfaceVpcEndpointStack(app, 'ProdPrivateLinkCartman', {
    env,
    vpc: vpcStack.vpc,
    domainName: CARTMAN_HOSTNAME,
    service: { name: 'com.amazonaws.vpce.us-west-2.vpce-svc-03ade4b2e790128c1', port: 443 },
  });

  const eventBusCfn = new EventBusStack(app, 'ProdEventBus', { env });

  const pdmsLambdaStack = new PdmsLambdaStack(app, 'ProdPdmsLambda', {
    env,
    vpc: vpcStack.vpc,
    eventBusCfn,
    eventBusSqsQueueName: 'ProdPdmsLambda-eventbus-queue',
    eventBusSqsDeadletterqueueName: 'ProdPdmsLambda-eventbus-deadletterqueue',
    s3Bucket: s3BucketStack.bucket,
    s3CodeFile: 'lambda_go_build.zip', // lambda needs something to start, this can be pushed with make upload-code-to-s3 in https://git-aws.internal.justin.tv/devhub/config-service-pdms-lambda
    authClientId: 'ixcntszjbdj6m0afu4wizfmyxhs2yq',
    authClientSecret: configServiceSecretsStack.pdmsLambdaClientSecret,
    owlHost: new cdk.StringConcat().join('https://', OWL_HOSTNAME),
    cartmanHost: new cdk.StringConcat().join('https://', CARTMAN_HOSTNAME),
    configServiceHost: new cdk.StringConcat().join('http://', CONFIG_SERVICE_HOSTNAME),
    configServiceId: CONFIG_SERVICE_CATALOG_ID,
    pdmsApiLambdaArn: 'arn:aws:lambda:us-west-2:125704643346:function:PDMSLambda-LambdaFunction-11LXHJVSNJJPY:live',
    pdmsAsumeRoleArn: 'arn:aws:iam::125704643346:role/PDMSLambda-CallerRole-13IIND444YKVR',
    alarmTopic: snsPagerdutyStack.lowUrgencyTopic,
  });

  new ConfigServiceStack(app, 'ProdConfigService', {
    datadogSecret: configServiceSecretsStack.datadog,
    env,
    networkLoadBalancerAttrs: {
      loadBalancerArn:
        'arn:aws:elasticloadbalancing:us-west-2:816111448886:loadbalancer/net/a6b93c16ecb6b11e8895206adaa5f84e/1378d10c4c822dc4',
      loadBalancerCanonicalHostedZoneId: 'Z18D5FSROUN65G',
      loadBalancerDnsName: 'a6b93c16ecb6b11e8895206adaa5f84e-1378d10c4c822dc4.elb.us-west-2.amazonaws.com',
    },
    prod: true,
    repository: configServiceReposStack.primary,
    sandstormRoleArn:
      'arn:aws:iam::734326455073:role/sandstorm/production/templated/role/extensions-configservice-prod',
    vpc: vpcStack.vpc,
    zone: zoneStack.zone,
  });

  new JenkinsUserStack(app, 'ProdJenkinsUser', {
    env,
    lambdaFunctionArn: pdmsLambdaStack.lambdaFn.functionArn,
    s3BucketArn: s3BucketStack.bucket.bucketArn,
  });
}
