import { IVpc, Peer, Port, SecurityGroup } from '@aws-cdk/aws-ec2';
import { IRepository } from '@aws-cdk/aws-ecr';
import {
  AwsLogDriver,
  Cluster,
  ContainerImage,
  FargateService,
  FargateTaskDefinition,
  Secret,
  UlimitName,
} from '@aws-cdk/aws-ecs';
import {
  NetworkLoadBalancer,
  NetworkLoadBalancerAttributes,
  NetworkTargetGroup,
  Protocol,
} from '@aws-cdk/aws-elasticloadbalancingv2';
import { Effect, PolicyStatement } from '@aws-cdk/aws-iam';
import { ARecord, IHostedZone, RecordTarget } from '@aws-cdk/aws-route53';
import { LoadBalancerTarget } from '@aws-cdk/aws-route53-targets';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, Duration, Stack, StackProps } from '@aws-cdk/core';

interface ConfigServiceStackProps extends StackProps {
  vpc: IVpc;
  /** If true, service will be scaled to production levels. */
  prod?: boolean;
  /** ECR repository containing the service. */
  repository: IRepository;
  /** ARN for Sandstorm role to assume. */
  sandstormRoleArn: string;
  /** Secret containing the Datadog API key. */
  datadogSecret: ISecret;
  /** Already existing NLB to add as listener and target for hosted zone root domain. */
  networkLoadBalancerAttrs: NetworkLoadBalancerAttributes;
  /** Route53 Hosted Zone for the DNS record targeting the network load balancer. */
  zone: IHostedZone;
}

/**
 * Config Service ECS cluster, service, load balancer, and DNS record.
 */
export class ConfigServiceStack extends Stack {
  constructor(scope: Construct, name: string, props: ConfigServiceStackProps) {
    super(scope, name, props);

    // ECS Cluster
    const cluster = new Cluster(this, 'Cluster', {
      vpc: props.vpc,
    });

    // ECS Task Definition
    const taskDefinition = new FargateTaskDefinition(this, 'TaskDef', {
      cpu: 1024, // 1 vCPU
      memoryLimitMiB: 2048, // 2 GB
    });
    taskDefinition.taskRole.addManagedPolicy({
      managedPolicyArn: `arn:aws:iam::${this.account}:policy/config_service_dynamo_policy`,
    });

    taskDefinition.taskRole.addManagedPolicy({
      managedPolicyArn: `arn:aws:iam::${this.account}:policy/config_service_sns_policy`,
    });

    taskDefinition.addToTaskRolePolicy(
      new PolicyStatement({
        actions: ['sts:AssumeRole'],
        resources: [props.sandstormRoleArn],
        effect: Effect.ALLOW,
      })
    );

    // ECS Container
    const container = taskDefinition.addContainer('Container', {
      image: ContainerImage.fromEcrRepository(props.repository, 'latest'),
      logging: new AwsLogDriver({ streamPrefix: 'ConfigService' }),
      dockerLabels: {
        'com.datadoghq.ad.check_names': '["config_service"]',
        'com.datadoghq.ad.init_configs': '[{}]',
        'com.datadoghq.ad.instances': '[{"host":"%%host%%","port":"8000"}]',
        'com.docker.compose.service': 'config_service',
      },
      environment: {
        'app.env': props.prod ? 'prod' : 'dev',
        'auth.method': 'cartman',
        'auth.cartman.secret_name': `identity/cartman/${props.prod ? 'production' : 'staging'}/ecc_public_key`,
        'auth.sandstorm.region': 'us-west-2',
        'auth.sandstorm.role_arn': props.sandstormRoleArn,
        'auth.sandstorm.table': 'sandstorm-production',
        'auth.sandstorm.key_id': 'alias/sandstorm-production',
        'events.sns.arn': `arn:aws:sns:us-west-2:${this.account}:extension-configuration-events`,
        'store.cache.duration': '1s',
        'stats.method': 'statsd',
        'stats.statsd.host': '127.0.0.1:8125',
        'stats.frequency.common': '1.0',
        'stats.frequency.frequent': '1.0',
        'stats.frequency.rare': '1.0',
      },
    });
    container.addPortMappings({ containerPort: 8000 });
    container.addUlimits({
      name: UlimitName.NOFILE,
      softLimit: 10240,
      hardLimit: 10240,
    });

    // Datadog Sidecar Container
    const datadog = taskDefinition.addContainer('datadog-agent', {
      cpu: 10,
      environment: {
        ECS_FARGATE: 'true',
        DD_DOCKER_LABELS_AS_TAGS: '{"com.docker.compose.service":"service_name"}',
        DD_DOGSTATSD_NON_LOCAL_TRAFFIC: 'true',
        DD_PROCESS_AGENT_ENABLED: 'true',
      },
      secrets: {
        DD_API_KEY: Secret.fromSecretsManager(props.datadogSecret),
      },
      image: ContainerImage.fromRegistry('datadog/agent:latest'),
      memoryReservationMiB: 256,
      logging: new AwsLogDriver({ streamPrefix: 'DataDog' }),
    });
    datadog.addPortMappings({ containerPort: 8125 });

    // ECS Service Security Group
    const serviceSecurityGroup = new SecurityGroup(this, 'ServiceSecurityGroup', {
      allowAllOutbound: true,
      vpc: props.vpc,
    });
    serviceSecurityGroup.addIngressRule(Peer.anyIpv4(), Port.tcp(8000));

    // ECS Service
    const service = new FargateService(this, 'Service', {
      cluster,
      desiredCount: props.prod ? 3 : 1,
      taskDefinition,
      minHealthyPercent: 100,
      maxHealthyPercent: 200,
      securityGroup: serviceSecurityGroup,
    });

    // ECS Service Auto Scaling
    const scaling = service.autoScaleTaskCount({
      minCapacity: props.prod ? 3 : 1,
      maxCapacity: props.prod ? 30 : 3,
    });

    scaling.scaleOnCpuUtilization('Cpu', {
      targetUtilizationPercent: 60,
    });

    scaling.scaleOnMemoryUtilization('Memory', {
      targetUtilizationPercent: 90,
    });

    // Add to Load Balancer
    const loadBalancer = NetworkLoadBalancer.fromNetworkLoadBalancerAttributes(
      this,
      'LoadBalancer',
      props.networkLoadBalancerAttrs
    );

    const targetGroup = new NetworkTargetGroup(this, 'TargetGroup', {
      port: 8000,
      deregistrationDelay: Duration.seconds(60),
      healthCheck: {
        path: '/debug/running',
        protocol: Protocol.HTTP,
      },
      vpc: props.vpc,
      targets: [service],
    });

    const listener = loadBalancer.addListener('Listener', {
      port: props.prod ? 81 : 80, // Port 81 for prod is temp
    });

    listener.addTargetGroups('TargetGroups', targetGroup);

    // DNS CNAME
    new ARecord(this, 'DnsRecord', {
      zone: props.zone,
      target: RecordTarget.fromAlias(new LoadBalancerTarget(loadBalancer)),
    });
  }
}
