import iam = require('@aws-cdk/aws-iam');
import cdk = require('@aws-cdk/core');

interface JenkinsUserStackProps extends cdk.StackProps {
  lambdaFunctionArn: string;
  s3BucketArn: string;
}

// Jenkins IAM user is defined here with CDK. After this stack is delployed, the
// credentials for Jenkins are generated in the AWS console (IAM, Security, Access key/secret pair),
// and then those are added to the credential store in Jenkins for the specific `<org>/<repo>`:
// https://jenkins.internal.justin.tv/job/<org>/job/<repo>/credentials/store/folder/domain/_/
// In this case, that is /job/devhub/job/config-service-pdms-lambda/credentials/store/folder/domain/_/
export class JenkinsUserStack extends cdk.Stack {
  constructor(scope: cdk.Construct, name: string, props: JenkinsUserStackProps) {
    super(scope, name, props);

    const jenkinsUser = new iam.User(this, 'JenkinsUser');

    // Global permissions
    jenkinsUser.addToPolicy(
      new iam.PolicyStatement({
        actions: ['tag:GetResources'],
        resources: ['*'],
        effect: iam.Effect.ALLOW,
      })
    );

    // Deploy PDMS lambda function code
    jenkinsUser.addToPolicy(
      new iam.PolicyStatement({
        actions: ['lambda:UpdateFunctionCode'],
        resources: [props.lambdaFunctionArn],
        effect: iam.Effect.ALLOW,
      })
    );

    // Put S3 file, for cdk deploys that may re-build the lambda from s3
    jenkinsUser.addToPolicy(
      new iam.PolicyStatement({
        actions: ['s3:PutObject'],
        resources: [props.s3BucketArn, 'arn:aws:s3:::*/*'], // any object on that bucket
        effect: iam.Effect.ALLOW,
      })
    );
  }
}
