import cloudwatch = require('@aws-cdk/aws-cloudwatch');
import ec2 = require('@aws-cdk/aws-ec2');
import iam = require('@aws-cdk/aws-iam');
import kms = require('@aws-cdk/aws-kms');
import lambda = require('@aws-cdk/aws-lambda');
import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
import s3 = require('@aws-cdk/aws-s3');
import secretsmanager = require('@aws-cdk/aws-secretsmanager');
import sns = require('@aws-cdk/aws-sns');
import sqs = require('@aws-cdk/aws-sqs');
import cdk = require('@aws-cdk/core');
import { EventBusStack } from './event-bus';

interface PdmsLambdaStackProps extends cdk.StackProps {
  vpc: ec2.IVpc;
  eventBusCfn: EventBusStack;
  eventBusSqsQueueName: string;
  eventBusSqsDeadletterqueueName: string;
  s3Bucket: s3.IBucket; // bucket where the code file is located
  s3CodeFile: string; // zip file for the lambda function in the s3Bucket
  authClientId: string;
  authClientSecret: secretsmanager.ISecret;
  owlHost: string;
  cartmanHost: string;
  configServiceHost: string;
  configServiceId: string; // from the service catalog, identifier to be used on PDMS
  pdmsApiLambdaArn: string; // Lambda ARN to identify the remote PDMS API
  pdmsAsumeRoleArn: string; // PDMSService:callerRole that is assumed when calling the remote PDMS API
  alarmTopic: sns.ITopic; // topic to report CloudWatch alarms
}

// PDMS lambda and SQS queues to handle EventBus notifications:
// Lambda code is in: https://git-aws.internal.justin.tv/devhub/config-service-pdms-lambda
export class PdmsLambdaStack extends cdk.Stack {
  public lambdaFn: lambda.IFunction;

  constructor(scope: cdk.Construct, name: string, props: PdmsLambdaStackProps) {
    super(scope, name, props);

    // KMS encryption key from the EventBus cfn template
    const key = kms.Key.fromKeyArn(this, 'EventBusKMSMasterKeyARN', props.eventBusCfn.kmsMasterKeyARN());

    // SQS Queue with dead letter queue
    if (!props.eventBusSqsQueueName.includes('eventbus')) {
      throw new Error(
        'PdmsLambdaStackProps.eventBusSqsQueueName must contain "eventbus" in lowercase for the EventBus to allow access'
      );
    }
    const deadLetterQueue = new sqs.Queue(this, 'EventBusDeadletterQueue', {
      queueName: props.eventBusSqsDeadletterqueueName,
      encryptionMasterKey: key,
      retentionPeriod: cdk.Duration.days(14), // max allowed, gives the team some time to fix the issue
    });
    const queue = new sqs.Queue(this, 'EventBusQueue', {
      queueName: props.eventBusSqsQueueName,
      encryptionMasterKey: key,
      deadLetterQueue: { queue: deadLetterQueue, maxReceiveCount: 5 }, // try 5 before sending to dead letter queue
      retentionPeriod: cdk.Duration.days(4),
    });

    // Alarm if messages are not being processed at all
    const alarm1 = new cloudwatch.Alarm(this, 'EventBusInactiveQueueAlarm', {
      alarmDescription:
        'EventBus messages are being ignored. ConfigService PDMS Lambda function is not triggering. Must be fixed for GDPR compliance',
      evaluationPeriods: 1,
      metric: deadLetterQueue.metricApproximateAgeOfOldestMessage(), // max seconds over 5 minutes
      threshold: 60 * 60 * 2, // 2 hours
      actionsEnabled: false, // TODO: enable when PDMS is implemented and use
    });
    alarm1.addAlarmAction({
      bind: () => ({ alarmActionArn: props.alarmTopic.topicArn }),
    });

    // Alarm if messages are repeatedly failing to be processed (deadLetterQueue has messages)
    const alarm2 = new cloudwatch.Alarm(this, 'EventBusDeadLetterQueueAlarm', {
      alarmDescription:
        'EventBus messages are causing errors on the ConfigService PDMS Lambda function. SQS DeadLetterQueue has those messages. Must be fixed for GDPR compliance.',
      evaluationPeriods: 1,
      metric: deadLetterQueue.metricApproximateNumberOfMessagesVisible(),
      threshold: 1,
      actionsEnabled: false, // TODO: enable when PDMS is implemented and use
    });
    alarm2.addAlarmAction({
      bind: () => ({ alarmActionArn: props.alarmTopic.topicArn }),
    });

    // SQS Policy from the EventBus cfn template to allow the event bus account to push events.
    // The sqs.QueuePolicy construct is usually better to define policies, but we have to use the lower level
    // sqs.CfnQueuePolicy construct instead, because the EventBus cfn template defines the policy in plain JSON.
    new sqs.CfnQueuePolicy(this, 'EventBusQueuePolicy', {
      policyDocument: props.eventBusCfn.sqsPolicyDocument(),
      queues: [queue.queueUrl],
    });

    // Lambda Function
    const lambdaFn = new lambda.Function(this, 'LambdaHandler', {
      vpc: props.vpc,
      runtime: lambda.Runtime.GO_1_X,
      handler: 'main',
      code: lambda.Code.fromBucket(props.s3Bucket, props.s3CodeFile), // needs to exist, even if it is ignored later
      timeout: cdk.Duration.seconds(10),
      environment: {
        CLIENT_ID: props.authClientId,
        CLIENT_SECRET_ARN: props.authClientSecret.secretArn,
        OWL_HOST: props.owlHost,
        CARTMAN_HOST: props.cartmanHost,
        CONFIG_SERVICE_HOST: props.configServiceHost,
        CONFIG_SERVICE_ID: props.configServiceId,
        PDMS_API_LAMDBA_ARN: props.pdmsApiLambdaArn,
        PDMS_ASUME_ROLE_ARN: props.pdmsAsumeRoleArn,
      },
      initialPolicy: [
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: ['secretsmanager:GetSecretValue'],
          resources: [props.authClientSecret.secretArn],
        }),
      ],
    });
    this.lambdaFn = lambdaFn;
    const executionRole = lambdaFn.role!;

    // Give the Lambda execution role permission to interact with the Event Bus.
    // => EventBus =[SQS policy]=> SQS Queue =[EventBus access policy]=> Lambda Function
    executionRole.addManagedPolicy({
      managedPolicyArn: props.eventBusCfn.accessPolicyARN(),
    });

    // Assume role to call the PDMS API
    if (props.pdmsAsumeRoleArn !== '') {
      executionRole.addToPolicy(
        new iam.PolicyStatement({
          actions: ['sts:AssumeRole'],
          resources: [props.pdmsAsumeRoleArn],
          effect: iam.Effect.ALLOW,
        })
      );
    }

    // Listen from the SQS Queue
    lambdaFn.addEventSource(new SqsEventSource(queue));
  }
}
