package clients

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"time"

	"code.justin.tv/foundation/twitchclient"
	cartman "code.justin.tv/web/cartman/client"
)

// Cartman generates Cartman Tokens to authorize capabilities.
type Cartman struct {
	cartman cartman.Client
}

func MustNewCartman(cartmanHost string) *Cartman {
	cartmanCli, err := cartman.NewClient(twitchclient.ClientConf{
		Host: cartmanHost,
	})
	if err != nil {
		log.Panicf("Could not init web/cartman client: %v", err)
	}
	return &Cartman{
		cartman: cartmanCli,
	}
}

func (c *Cartman) AuthorizeEditExtensionBroadcasterConfigs(ctx context.Context, now time.Time, oauthToken string, channelID string) (string, error) {
	tokenStr, err := c.getCartmanToken(ctx, now, oauthToken,
		[]string{"extensions::edit_broadcaster_configuration"},
		map[string]string{"channel_id": channelID}, // user_id of the channel where configs are going to be removed
	)
	if err != nil {
		return "", fmt.Errorf("AuthorizeEditExtensionBroadcasterConfigs(channelID: %q): %w", channelID, err)
	}
	return tokenStr, nil
}

// getCartmanToken is a wrapper method around the (not so readable) Cartman client method to get auth tokens.
func (c *Cartman) getCartmanToken(ctx context.Context, now time.Time, oauthToken string, capabilities []string, authFnParams map[string]string) (string, error) {
	origin, _ := http.NewRequest("GET", "/", nil)
	origin.Header.Set(cartman.AuthorizationHeader, "OAuth "+oauthToken)

	resp, err := c.cartman.GetAuthorizationToken(ctx, cartman.RequestParams{
		Origin:       origin, // OAuth token is read from the origin request and added as Authorization header
		KeyPath:      "ecc.key",
		Capabilities: capabilities,
		Params:       authFnParams, // extra params for the authorization functions as needed by the capabilities
	}, nil)
	if err != nil {
		return "", fmt.Errorf("cartman.GetAuthorizationToken: %w", err)
	}

	return resp.Token, nil
}
