package clients

import (
	"context"
	"errors"
	"net/http"
	"testing"
	"time"

	"github.com/stretchr/testify/require"

	"code.justin.tv/foundation/twitchclient"
	cartman "code.justin.tv/web/cartman/client"
)

func Test_Cartman__AuthorizeEditExtensionBroadcasterConfigs(t *testing.T) {
	ctx := context.Background()
	now := time.Now()
	userID := "123"

	t.Run("Success", func(t *testing.T) {
		fakeCli := &FakeCartmanCli{ReturnsToken: &cartman.TokenResponse{Token: "CartmanToken"}}
		c := &Cartman{cartman: fakeCli}
		cartmanToken, err := c.AuthorizeEditExtensionBroadcasterConfigs(ctx, now, "fakeOAuthToken", userID)
		require.NoError(t, err)
		require.Equal(t, "CartmanToken", cartmanToken)
	})
	t.Run("Error", func(t *testing.T) {
		fakeCli := &FakeCartmanCli{ReturnsError: errors.New("cartman is down")}
		c := &Cartman{cartman: fakeCli}
		_, err := c.AuthorizeEditExtensionBroadcasterConfigs(ctx, now, "fakeOAuthToken", userID)
		require.EqualError(t, err, "AuthorizeEditExtensionBroadcasterConfigs(channelID: \"123\"): cartman.GetAuthorizationToken: cartman is down")
	})
}

func Test_Cartman_getCartmanToken(t *testing.T) {
	ctx := context.Background()
	now := time.Now()

	t.Run("Success", func(t *testing.T) {
		fakeCli := &FakeCartmanCli{ReturnsToken: &cartman.TokenResponse{Token: "CartmanToken"}}
		c := &Cartman{cartman: fakeCli}
		cartmanToken, err := c.getCartmanToken(ctx, now, "fakeOAuthToken", []string{"fakeCapability"}, map[string]string{"fakeParamKey": "fakeParamVal"})
		require.NoError(t, err)
		require.Equal(t, "CartmanToken", cartmanToken)
	})
	t.Run("Error", func(t *testing.T) {
		fakeCli := &FakeCartmanCli{ReturnsError: errors.New("token is invalid")}
		c := &Cartman{cartman: fakeCli}
		_, err := c.getCartmanToken(ctx, now, "fakeOAuthToken", []string{"fakeCapability"}, map[string]string{"fakeParamKey": "fakeParamVal"})
		require.EqualError(t, err, "cartman.GetAuthorizationToken: token is invalid")
	})
}

//
// Test Helpers
//

type FakeCartmanCli struct {
	ReturnsToken *cartman.TokenResponse
	ReturnsError error
}

func (c *FakeCartmanCli) Authenticate(ctx context.Context, r *http.Request, reqOpts *twitchclient.ReqOpts) (*cartman.TokenResponse, error) {
	return c.ReturnsToken, c.ReturnsError
}
func (c *FakeCartmanCli) GetPermittedActions(ctx context.Context, rp cartman.RequestParams, reqOpts *twitchclient.ReqOpts) (*cartman.ActionsResponse, error) {
	return nil, c.ReturnsError
}
func (c *FakeCartmanCli) GetAuthorizationToken(ctx context.Context, rp cartman.RequestParams, reqOpts *twitchclient.ReqOpts) (*cartman.TokenResponse, error) {
	return c.ReturnsToken, c.ReturnsError
}
