package clients

import (
	"context"
	"fmt"
	"log"
	"time"

	"code.justin.tv/foundation/twitchclient"
	owl "code.justin.tv/web/owl/client"
)

// OwlApp authenticates an app identified by client ID and secret.
// The authenticated OAuth token is cached in memory and reused until it expires.
type OwlApp struct {
	clientID     string
	clientSecret string
	owl          owl.Client

	token Token
}

type Token struct {
	Val string
	Exp time.Time
}

func MustNewOwlApp(owlHost, clientID, clientSecret string) *OwlApp {
	owlCli, err := owl.NewClient(twitchclient.ClientConf{
		Host: owlHost,
	})
	if err != nil {
		log.Panicf("Could not init web/owl client: %v", err)
	}
	return &OwlApp{
		clientID:     clientID,
		clientSecret: clientSecret,
		owl:          owlCli,
		token:        Token{}, // zero value counts as already expired
	}
}

func (c *OwlApp) Authenticate(ctx context.Context, now time.Time) (string, error) {
	if c.token.Exp.After(now) {
		return c.token.Val, nil // cached
	}

	resp, err := c.owl.ExchangeClientCredentials(ctx, c.clientID, c.clientSecret, nil, nil)
	if err != nil {
		return "", fmt.Errorf("Authenticate: owl.ExchangeClientCredentials: %w", err)
	}

	c.token = Token{
		Val: resp.AccessToken,
		Exp: now.Add(time.Duration(resp.ExpiresIn-1) * time.Second),
	}

	return c.token.Val, nil
}
