package clients

import (
	"context"
	"errors"
	"testing"
	"time"

	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"

	owl "code.justin.tv/web/owl/client"
	owlmocks "code.justin.tv/web/owl/client/mocks"
)

func Test_OwlApp_Authenticate(t *testing.T) {
	ctx := context.Background()
	now := time.Now()

	t.Run("Uncached_Token", func(t *testing.T) {
		fakeCli := newFakeOwlClientThatReturns(&owl.ExchangeResponse{AccessToken: "REMOTE", ExpiresIn: 300}, nil)

		c := &OwlApp{owl: fakeCli, clientID: "fooClientID", clientSecret: "fooSecret"}
		oauthToken, err := c.Authenticate(ctx, now)
		require.NoError(t, err)
		require.Equal(t, "REMOTE", oauthToken)
	})
	t.Run("Cached_Token", func(t *testing.T) {
		fakeCli := newFakeOwlClientThatReturns(&owl.ExchangeResponse{AccessToken: "REMOTE", ExpiresIn: 300}, nil)
		c := &OwlApp{owl: fakeCli, clientID: "fooClientID", clientSecret: "fooSecret",
			token: Token{Val: "CACHED", Exp: now.Add(5 * time.Second)}} // non expired
		oauthToken, err := c.Authenticate(ctx, now)
		require.NoError(t, err)
		require.Equal(t, "CACHED", oauthToken)
	})
	t.Run("Cached_Expired_Token", func(t *testing.T) {
		fakeCli := newFakeOwlClientThatReturns(&owl.ExchangeResponse{AccessToken: "REMOTE", ExpiresIn: 300}, nil)
		c := &OwlApp{owl: fakeCli, clientID: "fooClientID", clientSecret: "fooSecret",
			token: Token{Val: "CACHED", Exp: now.Add(-5 * time.Second)}} // expired
		oauthToken, err := c.Authenticate(ctx, now)
		require.NoError(t, err)
		require.Equal(t, "REMOTE", oauthToken)
	})
	t.Run("Remote_Error", func(t *testing.T) {
		fakeCli := newFakeOwlClientThatReturns(nil, errors.New("owl is down"))
		c := &OwlApp{owl: fakeCli, clientID: "fooClientID", clientSecret: "fooSecret",
			token: Token{Val: "CACHED", Exp: now.Add(-5 * time.Second)}} // expired
		_, err := c.Authenticate(ctx, now)
		require.EqualError(t, err, "Authenticate: owl.ExchangeClientCredentials: owl is down")
	})
}

//
// Test Helpers
//

func newFakeOwlClientThatReturns(resp *owl.ExchangeResponse, err error) *owlmocks.Client {
	fakeCli := &owlmocks.Client{}
	fakeCli.On("ExchangeClientCredentials", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).
		Return(resp, err)
	return fakeCli
}
