package main

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"os"
	"time"

	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"

	"code.justin.tv/amzn/PDMSLambdaTwirp"
	"code.justin.tv/devhub/config-service-pdms-lambda/clients"
	eventbus "code.justin.tv/eventbus/client"
	"code.justin.tv/eventbus/client/subscriber/lambdafunc"
	"code.justin.tv/eventbus/schema/pkg/clock"
	"code.justin.tv/eventbus/schema/pkg/user"
	"code.justin.tv/eventbus/schema/pkg/user_data_request"
	configuration "code.justin.tv/extensions/configuration/services/main/client"
	"code.justin.tv/foundation/twitchclient"
)

type ConfEnv struct {
	ClientID          string
	ClientSecretArn   string
	OwlHost           string
	CartmanHost       string
	ConfigServiceHost string
	ConfigServiceID   string // from the service catalog, identifier to be used on PDMS
	PDMSAPILambdaARN  string // Lambda ARN to identify the remote PDMS API
	PDMSAsumeRoleARN  string // PDMSService:callerRole that is assumed when calling the remote PDMS API
}

func MustReadConfEnv() ConfEnv {
	return ConfEnv{
		ClientID:          mustReadEnv("CLIENT_ID"),
		ClientSecretArn:   mustReadEnv("CLIENT_SECRET_ARN"),
		OwlHost:           mustReadEnv("OWL_HOST"),
		CartmanHost:       mustReadEnv("CARTMAN_HOST"),
		ConfigServiceHost: mustReadEnv("CONFIG_SERVICE_HOST"),
		ConfigServiceID:   mustReadEnv("CONFIG_SERVICE_ID"),
		PDMSAPILambdaARN:  mustReadEnv("PDMS_API_LAMDBA_ARN"),
		PDMSAsumeRoleARN:  mustReadEnv("PDMS_ASUME_ROLE_ARN"),
	}
}

func main() {
	confEnv := MustReadConfEnv()
	log.SetFlags(0) // do not show Ldate, Ltime in the log message; CloudWatch already adds timestamps

	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Region:     aws.String("us-west-2"),
			HTTPClient: &http.Client{Timeout: 10 * time.Second},
		},
	}))

	secretsMngrCli := secretsmanager.New(sess)
	clientSecret := mustGetSecretValue(secretsMngrCli, confEnv.ClientSecretArn)

	authAppCli := clients.MustNewOwlApp(confEnv.OwlHost, confEnv.ClientID, clientSecret)
	authCartmanCli := clients.MustNewCartman(confEnv.CartmanHost)
	configServiceCli := clients.MustNewConfigServiceCli(confEnv.ConfigServiceHost)
	pdmsCli := clients.MustNewPDMSCli(sess, confEnv.PDMSAPILambdaARN, confEnv.PDMSAsumeRoleARN)

	// Request handler
	h := &handler{
		confEnv:          confEnv,
		authAppCli:       authAppCli,
		authCartmanCli:   authCartmanCli,
		configServiceCli: configServiceCli,
		pdmsCli:          pdmsCli,
	}
	mux := eventbus.NewMux()
	user.RegisterDestroyHandler(mux, h.UserDestroy)
	clock.RegisterUpdateHandler(mux, h.ClockUpdate)
	user_data_request.RegisterUserDataRequestCreateHandler(mux, h.UserDataRequestCreate)

	lambda.Start(lambdafunc.NewSQS(mux.Dispatcher()))
}

//
// EventBus event handlers
//

type handler struct {
	confEnv          ConfEnv
	authAppCli       *clients.OwlApp
	authCartmanCli   *clients.Cartman
	configServiceCli configuration.Client
	pdmsCli          PDMSLambdaTwirp.PDMSService
}

func (h *handler) UserDestroy(ctx context.Context, _ *eventbus.Header, event *user.Destroy) error {
	userID := event.UserId
	log.Printf("UserDestroy event received. UserID: %s", userID)
	now := time.Now()

	oauthToken, err := h.authAppCli.Authenticate(ctx, now)
	if err != nil {
		return fmt.Errorf("authAppCli.Authenticate.Error: %v", err)
	}

	cartmanToken, err := h.authCartmanCli.AuthorizeEditExtensionBroadcasterConfigs(ctx, now, oauthToken, userID)
	if err != nil {
		return fmt.Errorf("authCartmanCli.AuthorizeEditExtensionBroadcasterConfigs{userID: %q}.Error: %v", userID, err)
	}

	err = h.configServiceCli.DeleteChannelConfiguration(ctx, userID, &twitchclient.ReqOpts{AuthorizationToken: cartmanToken})
	if err != nil {
		return fmt.Errorf("configServiceCli.DeleteChannelConfiguration{userID: %q}.Error: %v", userID, err)
	}

	_, err = h.pdmsCli.ReportDeletion(ctx, &PDMSLambdaTwirp.ReportDeletionRequest{
		UserId:    userID,
		ServiceId: h.confEnv.ConfigServiceID,
	})
	if err != nil {
		return fmt.Errorf("pdmsCli.ReportDeletion{userID: %q, serviceID: %q}.Error: %v", userID, h.confEnv.ConfigServiceID, err)
	}

	log.Printf("UserDestroy event processed, ReportDeletion notified on PDMS. UserID: %s", userID)
	return nil
}

func (h *handler) UserDataRequestCreate(ctx context.Context, _ *eventbus.Header, event *user_data_request.UserDataRequestCreate) error {
	userID := event.UserId
	log.Printf("UserDataRequestCreate event received (ignored). UserID: %s", userID)
	// TODO: handle AccessFlow: https://wiki.twitch.com/display/SEC/The+Access+Flow
	//       That would be, authenticate the client token with Owl and Cartman,
	//       then call the ConfigService API to get all the configs for the channel (userID)
	//       then call pdmsCli.ReportUserAccessRequest with HasData: true if there's anything to report
	//       and then generate and upload the data in JSON to the provided S3 bucket link.
	// log.Printf("UserDataRequestCreate event processed, ReportUserAccessRequest notified on PDMS. UserID: %s", userID)
	return nil
}

func (h *handler) ClockUpdate(ctx context.Context, _ *eventbus.Header, event *clock.Update) error {
	// The ClockUpdate event can be used for manual testing by enabling the subscription in
	// the EventBus dashboard, then check if it was received in the lambda CloudWatch logs.
	log.Printf("ClockUpdate event received. Time.GetSeconds(): %d", event.Time.GetSeconds())
	return nil
}

//
// Helpers
//

func mustReadEnv(envVarName string) string {
	if val, ok := os.LookupEnv(envVarName); ok {
		return val
	}
	log.Panicf("os.LookupEnv: Missing environment variable: %v", envVarName)
	return ""
}

func mustGetSecretValue(secretsMngr *secretsmanager.SecretsManager, secretID string) string {
	result, err := secretsMngr.GetSecretValue(&secretsmanager.GetSecretValueInput{
		SecretId: aws.String(secretID),
		// VersionStage defaults to "AWSCURRENT"
	})
	if err != nil {
		log.Panicf("secretsMngr.GetSecretValue(SecretId: %q). Error: %v", secretID, err)
	}
	if result.SecretString == nil {
		log.Panicf("secretsMngr.GetSecretValue(SecretId: %q). result.SecretString is nil", secretID)
	}
	return *result.SecretString
}
