package protocol

import (
	"fmt"
	"net/http"

	"code.justin.tv/gds/gds/golibs/errors"
)

const (
	ErrConcurrencyCode            = "concurrency_failure"
	ErrChannelIDMismatchCode      = "channel_id_mismatch"
	ErrForbiddenByBroadcasterCode = "forbidden_by_broadcaster"
	ErrInvalidCacheKeyCode        = "invalid_cache_key"
	ErrInvalidCacheContentCode    = "invalid_cache_content"
	ErrInvalidCommonParameterCode = "invalid_common_parameter"
	ErrIllegalSegmentChannelCode  = "illegal_segment_channel"
	ErrIllegalSegmentLengthCode   = "illegal_segment_length"
	ErrMissingChannelIDCode       = "missing_channel_id"
	ErrMissingExtensionIDCode     = "missing_extension_id"
	ErrNonAtomicWriteCode         = "non_atomic_write"
	ErrUnknownSegmentTypeCode     = "unknown_segment_type"
)

var (
	ErrConcurrency            = errors.NewBuilder("Concurrency failure: please retry").WithErrorCode(ErrConcurrencyCode).WithHTTPStatus(http.StatusConflict).Build()
	ErrChannelIDMismatch      = errors.NewBuilder("Request scoped by Channel ID returned segments with different Channel IDs").WithErrorCode(ErrChannelIDMismatchCode).WithHTTPStatus(http.StatusInternalServerError).Build()
	ErrForbiddenByBroadcaster = errors.NewBuilder("Broadcaster has forbidden channel configuration").WithErrorCode(ErrForbiddenByBroadcasterCode).WithHTTPStatus(http.StatusForbidden).Build()
	ErrInvalidCacheKey        = errors.NewBuilder("Cache key was invalid").WithErrorCode(ErrInvalidCacheKeyCode).WithHTTPStatus(http.StatusServiceUnavailable).Build()
	ErrInvalidCacheContent    = errors.NewBuilder("Cache content was invalid").WithErrorCode(ErrInvalidCacheContentCode).WithHTTPStatus(http.StatusServiceUnavailable).Build()
	ErrInvalidCommonParameter = errors.NewBuilder("Query parameter Common must be a boolean").WithErrorCode(ErrInvalidCommonParameterCode).WithHTTPStatus(http.StatusBadRequest).Build()
	ErrMissingChannelID       = errors.NewBuilder("Channel ID is required").WithErrorCode(ErrMissingChannelIDCode).WithHTTPStatus(http.StatusUnprocessableEntity).Build()
	ErrMissingExtensionID     = errors.NewBuilder("Extension ID is required").WithErrorCode(ErrMissingExtensionIDCode).WithHTTPStatus(http.StatusUnprocessableEntity).Build()
	ErrNonAtomicWrite         = errors.NewBuilder("Request would result in a non-atomic write; rejected").WithErrorCode(ErrNonAtomicWriteCode).WithHTTPStatus(http.StatusUnprocessableEntity).Build()
	ErrTooManyRequests        = errors.ErrTooManyRequests
	ErrUnauthorized           = errors.ErrUnauthorized
	ErrUnavailable            = errors.ErrServiceUnavailable
	ErrUnimplemented          = errors.ErrNotImplemented
)

// ErrUnknownSegmentType constructs an error describing a bad segment
func ErrUnknownSegmentType(typ SegmentType) error { return &errUnknownSegmentType{typ} }

type errUnknownSegmentType struct {
	Type SegmentType
}

func (*errUnknownSegmentType) ErrorCode() string         { return ErrUnknownSegmentTypeCode }
func (*errUnknownSegmentType) HTTPStatus() int           { return http.StatusUnprocessableEntity }
func (e *errUnknownSegmentType) Details() errors.Details { return errors.Details{"type": e.Type} }
func (e *errUnknownSegmentType) Error() string {
	return fmt.Sprintf("Unknown segment type \"%v\"", e.Type)
}

type errIllegalSegmentChannel struct {
	Type    SegmentType
	Channel string
}

// ErrIllegalSegmentChannel constructs an error describing a bad segment
func ErrIllegalSegmentChannel(typ SegmentType, ch string) error {
	return &errIllegalSegmentChannel{typ, ch}
}

func (*errIllegalSegmentChannel) ErrorCode() string { return ErrIllegalSegmentChannelCode }
func (*errIllegalSegmentChannel) HTTPStatus() int   { return http.StatusUnprocessableEntity }
func (e *errIllegalSegmentChannel) Details() errors.Details {
	return errors.Details{"type": e.Type, "channel": e.Channel}
}
func (e *errIllegalSegmentChannel) Error() string {
	return fmt.Sprintf("Segment type \"%v\" is incompatible with channel \"%v\"", e.Type, e.Channel)
}

type errIllegalSegmentLength struct {
	Address Address
	Length  int
}

// ErrIllegalSegmentLength constructs an error describing a bad segment length
func ErrIllegalSegmentLength(a Address, length int) error {
	return &errIllegalSegmentLength{a, length}
}

func (*errIllegalSegmentLength) ErrorCode() string { return ErrIllegalSegmentLengthCode }
func (*errIllegalSegmentLength) HTTPStatus() int   { return http.StatusUnprocessableEntity }
func (e *errIllegalSegmentLength) Details() errors.Details {
	return errors.Details{
		"channel_id":   e.Address.ChannelID,
		"extension_id": e.Address.ExtensionID,
		"length":       e.Length,
		"max_length":   MaxSegmentLength,
		"type":         e.Address.SegmentType,
	}
}

func (e *errIllegalSegmentLength) Error() string {
	if e.Address.ChannelID == "" {
		return fmt.Sprintf("Segment \"%v\" length %v > maximum of %v for extension \"%v\"",
			e.Address.SegmentType,
			e.Length,
			MaxSegmentLength,
			e.Address.ExtensionID,
		)
	}
	return fmt.Sprintf("Segment \"%v\" length %v > maximum of %v for extension \"%v\", channel \"%v\"",
		e.Address.SegmentType,
		e.Length,
		MaxSegmentLength,
		e.Address.ExtensionID,
		e.Address.ChannelID,
	)
}
