package protocol

import "encoding/json"

// RecordMap is a mapping of addresses to records
type RecordMap map[Address]*Record

// MarshalJSON implements json.Marshaler
func (r RecordMap) MarshalJSON() ([]byte, error) {
	values := make(map[string]*Record)
	for k, v := range r {
		values[k.String()] = v
	}
	return json.Marshal(values)
}

// UnmarshalJSON implements json.Unmarshaler
func (r RecordMap) UnmarshalJSON(data []byte) error {
	values := make(map[string]*Record)
	if err := json.Unmarshal(data, &values); err != nil {
		return err
	}
	for k, v := range values {
		r[ParseAddress(k)] = v
	}
	return nil
}

type ConfigByExtension map[string]map[SegmentType]*Record

// RecordMapToChannelReturn must already be scoped by channel or this will error
func RecordMapToChannelReturn(data RecordMap) (ConfigByExtension, error) {
	extMap := make(map[string]map[SegmentType]*Record)
	if data != nil {
		channelID := ""
		for address, record := range data {
			if address.ChannelID != "" {
				if channelID == "" {
					channelID = address.ChannelID
				} else if channelID != address.ChannelID {
					return nil, ErrChannelIDMismatch
				}
			}
			extConfig, ok := extMap[address.ExtensionID]
			if ok {
				extConfig[address.SegmentType] = record
			} else {
				extMap[address.ExtensionID] = map[SegmentType]*Record{address.SegmentType: record}
			}
		}
	}
	return extMap, nil
}
