package cartman

import (
	"context"
	"net/http"

	"code.justin.tv/foundation/twitchclient"
)

// TokenResponse defines the format of the response for authorization token
// requests
type TokenResponse struct {
	Token       string `json:"token"`
	UserID      string `json:"user_id,omitempty"`
	ClientID    string `json:"client_id,omitempty"`
	ClientRowID string `json:"client_row_id,omitempty"`
}

// Token defines the expected structure of an authorization token from
// Cartman
type Token string

const (
	tokenRequest = "authorization_token"
)

// GetAuthorizationToken builds a request to Cartman's /authorization_token
// endpoint and returns the authorization token returned in the response of
// that request
func (c *client) GetAuthorizationToken(ctx context.Context, rp RequestParams, reqOpts *twitchclient.ReqOpts) (*TokenResponse, error) {
	req, err := c.buildRequest(ctx, rp, tokenRequest)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cartman.get_authorization_token",
		StatSampleRate: defaultStatSampleRate,
	})

	var resp TokenResponse
	_, err = c.DoJSON(ctx, &resp, req, combinedReqOpts)
	return &resp, err
}

// Authenticate builds a request to Cartman's /authenticate endpoint and
// returns the authorization token returned in the response of that request
func (c *client) Authenticate(ctx context.Context, r *http.Request, reqOpts *twitchclient.ReqOpts) (*TokenResponse, error) {
	req, err := c.NewRequest("GET", "/authenticate", nil)
	if err != nil {
		return nil, err
	}

	addAuthentications(ctx, req.Header, r)
	addSecurityTokens(req, r)

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cartman.authenticate",
		StatSampleRate: defaultStatSampleRate,
	})

	var resp TokenResponse
	_, err = c.DoJSON(ctx, &resp, req, combinedReqOpts)
	return &resp, err
}
