// Load required packages.
const { copyFileSync, existsSync, mkdirSync, readdirSync, unlinkSync } = require('fs');
const { join } = require('path');
const { createVersion, compareVersions } = require('./public/version');

// Create or empty the 'next' directory.
if (existsSync('next')) {
  for (const fileName of readdirSync('next')) {
    unlinkSync(join('next', fileName));
  }
} else {
  mkdirSync('next');
}

// Determine the latest version.
const nullVersion = createVersion('0.0.0');
const versionMatches = readdirSync('dist').
  map((fileName) => /^(.*)([0-9]+\.[0-9]+\.[0-9]+)(.*)$/.exec(fileName)).
  filter((matches) => matches).
  map((matches) => ({ matches, version: createVersion(matches[2]) }));
const selectedVersion = versionMatches.map((vm) => vm.version).reduce((p, c) => compareVersions(p, c) < 0 ? c : p, nullVersion);
if (compareVersions(selectedVersion, nullVersion) === 0) {
  console.error('cannot determine latest version');
  process.exit(1);
}

// Determine the latest installer files.
const selectedFilePairs = versionMatches.filter((vm) => compareVersions(selectedVersion, vm.version) === 0).map(({ matches }) => {
  let prefix = matches[1];
  while ([' ', '.', '-', '_'].some((ch) => prefix.endsWith(ch))) {
    prefix = prefix.substr(0, prefix.length - 1);
  }
  return { source: matches[0], target: prefix + matches[3] };
});
if (!selectedFilePairs.length) {
  console.error('cannot determine latest installer files');
  process.exit(1);
}

// Create version-independent installer files and copy current installer files
// to the 'next' directory.
for (const selectedFilePair of selectedFilePairs) {
  const source = join('dist', selectedFilePair.source);
  const targets = [join('next', selectedFilePair.source), join('next', selectedFilePair.target)];
  targets.forEach((target) => copyFileSync(source, target));
}
if (process.platform === 'win32') {
  copyFileSync(join('dist', 'latest.yml'), join('next', 'latest.yml'));
} else {
  ['linux', 'mac'].forEach((platform) => {
    const target = `latest-${platform}.yml`;
    copyFileSync(join('dist', target), join('next', target));
  });
}
