const { basename } = require('path');
const https = require('https');
const btoa = require('js-base64').Base64.encode;
const { app } = require('electron');
const Store = require('electron-store');
const api = require('./api');
const { isDebuggingProduction } = require('./debug');

let extension, version;
if (process.argv[1] && basename(process.argv[1]) === 'analytics.test.js') {
  extension = {
    extension_id: process.argv[2],
    extension_version: process.argv[3],
    twitch_id: process.argv[4],
  };
  version = process.argv[5];
} else {
  if (isDebuggingProduction()) {
    debugger;
  }

  api.add('/event', sendEvent);

  extension = (() => {
    const store = new Store();
    const extension = store.get('extension') || {};
    return {
      extension_id: extension.extension_id || '',
      extension_version: extension.extension_version || '',
      twitch_id: extension.twitch_id || '',
    };
  })();
  version = app.getVersion();
}

module.exports = {
  sendEvent,
  setExtension,
};

function setExtension(userId, clientId, version) {
  const store = new Store();
  Object.assign(extension, { extension_id: clientId, extension_version: version, twitch_id: userId });
  store.set('extension', extension);
}

async function sendEvent({ eventName, eventData }) {
  try {
    const data = {
      event: eventName,
      properties: {
        ...extension,
        ...eventData,
        time: new Date().toLocaleString(),
        version,
      },
    };
    const json = JSON.stringify(data);
    const body = `data=${btoa(json)}`;
    const headers = {
      'Accept': 'application/vnd.twitchtv.v5+json; charset=UTF-8',
      'Content-Length': body.length,
      'Content-Type': 'application/json; charset=UTF-8',
    };
    const method = 'POST';
    const request = {
      body,
      headers,
      method,
    };
    const response = await fetch('https://spade.twitch.tv', request);
    if (response.status >= 400) {
      throw new Error(response.data || `Cannot access Spade API; status ${response.status}`);
    } else if (response.status !== 204) {
      return response.data;
    }
  } catch (ex) {
    console.error('sendEvent failed', ex.message);
  }
}

async function fetch(url, options) {
  return await new Promise((resolve, reject) => {
    const request = https.request(url, options, (response) => {
      let data = '';
      response.setEncoding('utf8');
      response.on('data', (chunk) => {
        data += chunk;
      });
      response.on('end', () => {
        resolve({ data, status: response.statusCode });
      });
    });

    request.on('error', (ex) => {
      console.error('https.request failed', ex.message);
      reject(ex);
    });
    request.write(options.body);
    request.end();
  });
}
