const childProcess = require('child_process');
const { app, BrowserWindow, dialog, session, shell } = require('electron');
const electronLog = require('electron-log');
const Store = require('electron-store');
const { autoUpdater } = require('electron-updater');
const { existsSync, readdirSync, readFileSync } = require('fs');
const { basename, join } = require('path');
const { format, parse } = require('url');
const { sendEvent, setExtension } = require('./analytics');
const api = require('./api');
const { isDebuggingProduction } = require('./debug');
const menu = require('./menu');
const project = require('./project');
const twitch = require('./twitch');
const { createVersion } = require('./version');

if (isDebuggingProduction()) {
  debugger;
}

const nodeVersionResult = childProcess.spawnSync('node', ['--version']);
if (nodeVersionResult.error) {
  if (process.platform !== 'win32') {
    const nodeDirectory = findNodeDirectory();
    if (nodeDirectory) {
      process.env['PATH'] += ':' + nodeDirectory;
      delete nodeVersionResult.error;
    }
  }
}

api.add('/has-node', () => Promise.resolve(!nodeVersionResult.error));

const version = app.getVersion();

autoUpdater.autoDownload = true;
autoUpdater.logger = electronLog;
autoUpdater.logger.transports.console.level = 'info';
autoUpdater.logger.transports.file.level = false;
electronLog.info('App starting...');

project.setApplicationUrl(composeApplicationUrl());

const vsCodePath = process.platform === 'win32' ?
  join(process.env['LOCALAPPDATA'] || '', 'Programs', 'Microsoft VS Code', 'bin', 'code.cmd') :
  '/Applications/Visual Studio Code.app/Contents/MacOS/Electron';
api.add('/vs-code/check', () => {
  return new Promise((resolve, reject) => {
    try {
      resolve(existsSync(vsCodePath));
    } catch (ex) {
      reject(ex);
    }
  });
});

api.add('/vs-code/launch', ({ projectFolderPath }) => {
  const command = process.platform === 'win32' ? `"${vsCodePath}"` : vsCodePath;
  const child = childProcess.spawn(command, ['.'], {
    cwd: projectFolderPath,
    detached: true,
    shell: process.platform === 'win32',
    stdio: 'ignore',
  });
  child.unref();
});

let resolveAuth;
api.add('/auth', () => {
  return new Promise((resolve, _reject) => {
    resolveAuth = resolve;
  });
});

const contextMenu = menu.createContextMenu();
api.add('/context', () => (contextMenu.popup({ window: mainWindow }), undefined));

let onUpdateCheckComplete, updateCheckComplete;
api.add('/update', () => {
  return updateCheckComplete ? Promise.resolve() : new Promise((resolve, _reject) => {
    onUpdateCheckComplete = resolve;
  });
});
api.add('/dialog', async (options) => {
  const rv = await dialog.showOpenDialog(mainWindow, options);
  if (rv) {
    if (Array.isArray(rv)) {
      return rv[0];
    }
    return rv.filePaths && rv.filePaths[0];
  }
});

api.add('/message-box', async (options) => {
  options.buttons = options.buttons || [];
  options.title = options.title || 'Twitch Developer Rig';
  options.type = options.type || 'info';
  const rv = await dialog.showMessageBox(mainWindow, options);
  return rv.response;
});

api.add('/open-url', ({ url }) => {
  return shell.openExternal(url);
});

let wantsHttp;
api.add('/allow-http', (options) => {
  wantsHttp = options.wantsHttp;
});

let certificateCallback = () => { };
const certificateExceptions = [];
api.add('/certificate-exception/add', (certificateException) => {
  certificateExceptions.push(certificateException);
  certificateCallback(certificateException.isAllowed);
  certificateCallback = () => { };
});

const allowedUrls = [];
api.add('/set-project', ({ userId, project }) => {
  certificateExceptions.splice(0, certificateExceptions.length, ...project.certificateExceptions);
  wantsHttp = project.allowHttpBackend;
  const allowedConfigUrls = createUrls(project.manifest.whitelistedConfigUrls);
  const allowedPanelUrls = createUrls(project.manifest.whitelistedPanelUrls);
  allowedUrls.splice(0, allowedUrls.length, ...allowedConfigUrls, ...allowedPanelUrls);
  console.log('allowed URLs: ', ...allowedUrls.map((url) => url.href));
  setExtension(userId, project.manifest.id, project.manifest.version);

  function createUrls(urls) {
    return (urls || []).map(createUrl).filter((url) => url);

    function createUrl(url) {
      try {
        return parse(url);
      } catch (ex) {
        console.warn(`warning:  malformed URL ${url}; ignoring`);
      }
    }
  }
});

// Keep a global reference to the window object to prevent it from being
// destroyed during garbage collection.
let mainWindow;

// Configure the authorization redirect.  There is a similar mechanism here:
// https://www.manos.im/blog/electron-oauth-with-github/
process.env['RIG_CLIENT_ID'] = '07o96yph8nxna6znp32zgdxj0os5l5';
process.env['RIG_REDIRECT_URL'] = 'http://developer-rig';
// TODO:  see https://electronjs.org/docs/api/protocol for a possible custom
// protocol solution.
const redirectUrl = parse(process.env['RIG_REDIRECT_URL']);
let proxyUrl;

async function createFirstWindow() {
  // Start an internal Web server to serve Twitch content.
  proxyUrl = await twitch();

  // Check for an update.
  autoUpdater.on('update-downloaded', (event) => {
    electronLog.info('update-downloaded', event);
    autoUpdater.quitAndInstall();
  });
  autoUpdater.on('update-not-available', (event) => {
    electronLog.info('update-not-available', event);
    fireUpdateCheckComplete();
  });
  autoUpdater.on('error', (event) => {
    electronLog.error('error', event);
    fireUpdateCheckComplete();
  });
  autoUpdater.checkForUpdates();

  // Create the application menu.
  menu.createMenu(app, showAbout, createProject, openProject);

  // Create the main window.
  createWindow();

  api.add('/clear-auth', () => {
    return session.defaultSession.clearStorageData({ storages: ['cookies'] });
  });

  // Override the coordinator and OAuth redirect requests.
  const skipLog = ['client-event-reporter.twitch.tv', '.scorecardresearch.com', '.ttvnw.net'];
  session.defaultSession.webRequest.onBeforeRequest((details, callback) => {
    // Canceling a request here does not prevent a new window from opening if
    // the client requested one, such as for target="_blank" or window.open.
    // Such windows end up with about:blank as their content.
    const url = parse(details.url);
    if (!skipLog.some((s) => url.hostname && url.hostname.endsWith(s))) {
      electronLog.info(url.href);
    }
    if (url.pathname && basename(url.pathname) === 'player' && getAuthority(url) !== proxyUrl) {
      callback({ redirectURL: proxyUrl + '/player' + url.search });
    } else if (url.pathname && basename(url.pathname) === 'coordinator.js' && getAuthority(url) !== proxyUrl) {
      callback({ redirectURL: proxyUrl + '/coordinator.js' });
    } else if (url.protocol === redirectUrl.protocol && url.host === redirectUrl.host && url.hash) {
      resolveAuth(url.hash.substr(1));
      resolveAuth = () => { };
      callback({ redirectURL: 'about:blank' });
    } else {
      callback({});
    }
  });

  // Specify the Developer Rig as the source of the request.
  session.defaultSession.webRequest.onBeforeSendHeaders((details, callback) => {
    details.requestHeaders['X-Requested-With'] = `developer-rig; ${version}`;
    callback({ requestHeaders: details.requestHeaders });
  });

  // Modify the content security policy if not in local test.
  session.defaultSession.webRequest.onHeadersReceived((details, callback) => {
    const response = {};
    const hostname = parse(details.url).hostname;
    if (hostname && hostname.endsWith('.ext-twitch.tv')) {
      const responseHeaders = { ...details.responseHeaders };
      const contentSecurityPolicies = responseHeaders['content-security-policy'];
      if (contentSecurityPolicies) {
        let contentSecurityPolicy = contentSecurityPolicies[0];
        let frameAncestorsReplacement = `frame-ancestors file: ${proxyUrl}`;
        if (process.env['DEV_URL']) {
          frameAncestorsReplacement += ` ${process.env['DEV_URL']}`;
        }
        contentSecurityPolicy = contentSecurityPolicy.replace('frame-ancestors', frameAncestorsReplacement);
        if (wantsHttp) {
          contentSecurityPolicy = contentSecurityPolicy.replace('connect-src', 'connect-src http:');
        }
        responseHeaders['content-security-policy'] = [contentSecurityPolicy];
        response.responseHeaders = responseHeaders;
      }
    }
    callback(response);
  });

  sendEvent({ eventName: 'dx_rig_on_open', eventData: {} });
}

function createWindow() {
  // Create the browser window.
  const store = new Store();
  const options = store.get('bounds') || { width: 1152, height: 864 };
  options.webPreferences = { preload: join(__dirname, 'preload.js'), webviewTag: true };
  mainWindow = new BrowserWindow(options);

  // Load the index.html of the app.
  mainWindow.loadURL(composeApplicationUrl());

  // Open the DevTools.
  if (isDebuggingProduction()) {
    mainWindow.webContents.openDevTools();
  }

  // Full-screen or maximize the window if it was previously in either mode.
  if (store.get('is-full-screen')) {
    mainWindow.setFullScreen(true);
  } else if (store.get('is-maximized')) {
    mainWindow.maximize();
  }

  mainWindow.on('close', () => {
    const store = new Store();
    const bounds = mainWindow.getNormalBounds();
    store.set('bounds', bounds);
    if (process.platform === 'darwin') {
      store.set('is-full-screen', mainWindow.isFullScreen());
    }
    store.set('is-maximized', mainWindow.isMaximized());
  });

  mainWindow.on('closed', () => {
    if (process.platform !== 'darwin') {
      quit();
    }
    mainWindow = undefined;
  });

  mainWindow.webContents.on('new-window', (event, url) => {
    if (isAllowedUrl(parse(url))) {
      shell.openExternal(url);
    }
    event.preventDefault();

    function isAllowedUrl(url) {
      if (allowedUrls.some((allowedUrl) => url.host === allowedUrl.host)) {
        console.log(`allowing URL ${url.href}`);
        return true;
      }
      return url.host && url.host.endsWith('.twitch.tv');
    }
  });
}

function composeApplicationUrl() {
  const url = process.env['DEV_URL'] || format({
    protocol: 'file',
    pathname: join(__dirname, '..', 'build', 'index.html'),
  });
  return url;
}

// This method will be called when Electron has finished initialization and is
// ready to create browser windows.  Some APIs can only be used after this
// event occurs.
app.on('ready', createFirstWindow);

// Quit when all windows are closed.
app.on('window-all-closed', () => {
  // On OS X, it is common for applications and their menu bar to stay active
  // until the user quits explicitly with Cmd+Q.
  if (process.platform !== 'darwin') {
    quit();
  }
});

app.on('activate', () => {
  // On OS X, it is common to re-create a window in the app when the dock icon
  // is clicked and there are no other windows open.
  if (mainWindow === undefined) {
    createWindow();
  }
});

app.on('certificate-error', (event, _webContents, url, error, certificate, callback) => {
  url = new URL(url);
  const parts = url.hostname.split('.');
  if (parts[parts.length - 2] === 'scorecardresearch' && parts[parts.length - 1] === 'com') {
    // These are from the player.  Reject without asking.
    return;
  }
  const certificateException = certificateExceptions.find((e) => e.isOrigin ? url.origin === e.value : url.href === e.value);
  if (!certificateException) {
    event.preventDefault();
    certificateCallback = callback;
    askForCertificateException(url.href, error, certificate);
  } else if (certificateException.isAllowed) {
    event.preventDefault();
    callback(true);
  }
});

let isQuitting;
app.on('will-quit', (event) => {
  if (!isQuitting) {
    event.preventDefault();
    quit();
  }
});

async function quit() {
  if (!isQuitting) {
    isQuitting = true;
    certificateCallback = () => { };
    await project.shutdown();
    await sendEvent({ eventName: 'dx_rig_on_close', eventData: {} });
    app.quit();
  }
}

function fireUpdateCheckComplete() {
  updateCheckComplete = true;
  if (onUpdateCheckComplete) {
    onUpdateCheckComplete();
    onUpdateCheckComplete = undefined;
  }
}

function showAbout() {
  const options = {
    buttons: [],
    message: `Twitch Developer Rig v${version}`,
    title: 'Twitch Developer Rig',
    type: 'info',
  };
  dialog.showMessageBox(mainWindow, options);
}

function createProject() {
  mainWindow.webContents.send('main', { action: 'create project' });
}

function openProject() {
  mainWindow.webContents.send('main', { action: 'open project' });
}

function askForCertificateException(url, error, certificate) {
  mainWindow.webContents.send('main', { action: 'check certificate', exceptionRequest: { url, error, certificate } });
}

function findNodeDirectory() {
  const possibleDirectories = ['/usr/local/bin', '/usr/bin'];

  // Check for nvm.
  const home = process.env['HOME'] || '';
  const nvmDirectoryPath = process.env['NVM_DIR'] || join(home, '.nvm');
  if (existsSync(nvmDirectoryPath)) {
    const currentDirectoryPath = join(nvmDirectoryPath, 'current');
    const defaultFilePath = join(nvmDirectoryPath, 'alias', 'default');
    const nvmrcFilePath = join(home, '.nvmrc');
    const nvmNodeDirectoryPath = join(nvmDirectoryPath, 'versions', 'node');
    if (existsSync(nvmNodeDirectoryPath)) {
      const installedVersions = readdirSync(nvmNodeDirectoryPath).map(createVersion);
      if (installedVersions.length) {
        if (existsSync(currentDirectoryPath)) {
          possibleDirectories.unshift(join(currentDirectoryPath, 'bin'));
        } else if (existsSync(nvmrcFilePath)) {
          const nvmrcVersion = parseNvmVersion(nvmrcFilePath);
          possibleDirectories.unshift(makePath(nvmNodeDirectoryPath, nvmrcVersion.selectVersion(installedVersions)));
        } else if (existsSync(defaultFilePath)) {
          const defaultVersion = parseNvmVersion(defaultFilePath);
          possibleDirectories.unshift(makePath(nvmNodeDirectoryPath, defaultVersion.selectVersion(installedVersions)));
        }
      }
    }
  }

  // Determine if node is in a known place.
  return possibleDirectories.find((possibleDirectory) => {
    const result = childProcess.spawnSync(`${possibleDirectory}/node`, ['--version']);
    return !result.error;
  });

  function makePath(nvmNodeDirectoryPath, version) {
    return join(nvmNodeDirectoryPath, version.toString(), 'bin');
  }

  function parseNvmVersion(versionFilePath) {
    const versionText = readFileSync(versionFilePath, 'utf8');
    return createVersion(versionText);
  }
}

function getAuthority(url) {
  return url.href.substr(0, url.href.indexOf(url.pathname || ''));
}
