const { BrowserWindow } = require('electron');
const electronLog = require('electron-log');
const childProcess = require('child_process');
const fs = require('fs');
const { isAbsolute, join, resolve } = require('path');
const request = require('request');
const unzip = require('unzip');
const { sendEvent } = require('./analytics');
const api = require('./api');
const { readFile } = require('./async');
const { isDebuggingProduction } = require('./debug');
const host = require('./host');

if (isDebuggingProduction()) {
  debugger;
}

const children = {
  backend: undefined,
  frontend: undefined,
};
const examples = [
  {
    title: 'Getting Started',
    description: 'Click a button to change the color on a circle.',
    repository: 'twitchdev/extension-getting-started',
    frontendFolderName: join('extension-getting-started', 'public'),
    frontendCommand: '',
    isGuide: true,
    backendFolderName: join('extension-getting-started', 'services'),
    backendCommand: `node backend -c "{clientId}" -s "{secret}"`,
    npm: ['i'],
    expectedDuration: 8,
  },
  {
    title: 'Hello World + PubSub',
    description: 'Click a button to change the color on a circle for all viewers.',
    repository: 'twitchdev/extensions-hello-world',
    frontendFolderName: join('extensions-hello-world', 'public'),
    frontendCommand: '',
    isGuide: true,
    backendFolderName: join('extensions-hello-world', 'services'),
    backendCommand: `node backend -c "{clientId}" -s "{secret}" -o "{ownerId}"`,
    npm: ['i'],
    expectedDuration: 8,
  },
  {
    title: 'Boilerplate',
    description: 'Front-end-only React-based example to get you started building quickly.',
    repository: 'twitchdev/extensions-boilerplate',
    frontendFolderName: 'extensions-boilerplate',
    frontendCommand: 'npm run start',
    isGuide: false,
    npm: ['i'],
    expectedDuration: 51,
  },
  {
    title: 'Bot Commander',
    description: 'Front-end-only React-based example that leverages Configuration Service.',
    repository: 'twitchdev/bot-commander',
    frontendFolderName: 'bot-commander',
    frontendCommand: 'npm run start',
    isGuide: true,
    npm: ['i'],
    expectedDuration: 51,
  },
  {
    title: 'Animal Facts (requires Go 1.10+)',
    description: 'Front-end React-based example with back-end service that leverages Configuration Service.',
    repository: 'twitchdev/animal-facts',
    frontendFolderName: 'animal-facts',
    frontendCommand: 'npm run start',
    isGuide: true,
    backendFolderName: join('animal-facts', 'ebs'),
    backendCommand: process.platform === 'win32' ? 'ebs' : './ebs',
    npm: ['i'],
    expectedDuration: 51,
  },
];
const functions = {
  '/examples': getExamples,
  '/backend': startBackend,
  '/frontend': hostFrontend,
  '/project': createProject,
  '/status': getStatus,
  '/stop': stop,
  '/load': loadFile,
  '/save': saveFile,
};
const StopOptions = {
  Backend: 1,
  Frontend: 2,
};

Object.keys(functions).forEach((key) => {
  api.add(key, functions[key]);
});

module.exports = { getPort, setApplicationUrl, shutdown };

let applicationUrl, secondaryWindow;

async function getPort() {
  const status = await getStatus();
  return status.port;
}

function setApplicationUrl(applicationUrl_) {
  applicationUrl = applicationUrl_;
}

async function shutdown() {
  // Stop the back- and front-end services.
  await stop({ stopOptions: StopOptions.Backend | StopOptions.Frontend });

  // Destroy the secondary window if it is open.
  secondaryWindow && secondaryWindow.destroy();
}

async function getExamples() {
  return examples.map((example, index) => ({ ...example, id: index }));
}

async function setSecondaryWindow(sourceName) {
  // If there isn't a secondary window, open one and await its opening.
  if (!secondaryWindow) {
    const webPreferences = { preload: join(__dirname, 'preload.js') };
    const options = { autoHideMenuBar: true, width: 576, height: 432, show: false, webPreferences };
    secondaryWindow = new BrowserWindow(options);
    secondaryWindow.loadURL(applicationUrl + '?view=secondary');
    await new Promise((resolve, _reject) => {
      secondaryWindow.once('ready-to-show', () => {
        secondaryWindow.show();
        resolve();
      });
      secondaryWindow.once('closed', () => {
        secondaryWindow = undefined;
      });
    });
  }
  return {
    onStderr: (data) => secondaryWindow && secondaryWindow.webContents.send('stderr', sourceName, data),
    onStdout: (data) => secondaryWindow && secondaryWindow.webContents.send('stdout', sourceName, data),
  };
}

async function startBackend({ backendCommand, workingFolderPath }) {
  // Send output of the back end to the back-end panel of the secondary window.
  const redirections = await setSecondaryWindow('back-end');
  const args = parseCommandLine(backendCommand);
  const options = {};
  if (workingFolderPath) {
    options.cwd = workingFolderPath;
  }
  const command = args.shift();
  const { child, exitCode } = await spawn({ command, args, options, redirections });
  if (exitCode) {
    throw new Error(`Back-end command exited with exit code ${exitCode}`);
  }
  children.backend = child;
}

async function hostFrontend({ frontendCommand, frontendFolderPath, port, projectFolderPath }) {
  // Send output of the front end to the front-end panel of the secondary window.
  const redirections = await setSecondaryWindow('front-end');
  let args = [];
  const options = {};

  if (frontendCommand) {
    args = parseCommandLine(frontendCommand);
    const command = args.shift();
    if (isAbsolute(frontendFolderPath)) {
      options.cwd = frontendFolderPath;
    } else if (projectFolderPath) {
      options.cwd = resolve(projectFolderPath, frontendFolderPath);
    }
    if (process.platform === 'win32') {
      options.shell = true;
    }
    const { child, exitCode } = await spawn({ command, args, options, redirections });
    if (exitCode) {
      throw new Error(`Front-end command exited with exit code ${exitCode}`);
    }
    children.frontend = child;
  } else {
    children.frontend = await host(isAbsolute(frontendFolderPath) ? frontendFolderPath : resolve(projectFolderPath, frontendFolderPath), port, redirections);
    children.frontend.exitCode = null;
  }
}

async function createProject({ codeGenerationOption, exampleId, project }) {
  electronLog.info('Starting project creation');
  const { projectFolderPath } = project;
  delete project.projectFolderPath;
  delete project.secret;
  const rx = process.platform === 'win32' ? /[\\\/:\*\?"<>\|]+/g : /\/+/g;
  const projectFileName = project.name.replace(rx, '_') + '.json';
  const projectFilePath = resolve(projectFolderPath, projectFileName);
  const errorPrefix = `Invalid project folder "${projectFolderPath}"`;
  if (!isAbsolute(projectFolderPath)) {
    throw new Error(`${errorPrefix}; it must be an absolute path`);
  }
  if (!fs.existsSync(projectFolderPath)) {
    fs.mkdirSync(projectFolderPath);
  } else if (codeGenerationOption !== 'none' && ~fs.readdirSync(projectFolderPath).indexOf(projectFileName)) {
    throw new Error(`${errorPrefix}; it already has a file named "${projectFileName}"`);
  }
  let code_used = '';
  if (codeGenerationOption === 'example' && exampleId !== undefined) {
    const example = examples[exampleId];
    if (example && example.repository) {
      const { repository, npm } = example;
      const exampleFolderName = repository.split('/')[1];
      const exampleFolderPath = resolve(projectFolderPath, exampleFolderName);
      if (fs.existsSync(exampleFolderPath)) {
        throw new Error(`${errorPrefix}; it already has a sub-folder named "${exampleFolderName}"`);
      }
      await fetchExample(repository, projectFolderPath);
      // If necessary, run npm.
      if (npm) {
        const { exitCode, errorText } = await spawn({
          command: 'npm',
          args: npm,
          options: {
            cwd: exampleFolderPath,
            shell: true,
          }
        });
        if (exitCode) {
          throw new Error(`npm failed with exit code ${exitCode}: ${errorText}`);
        }
      }
      code_used = example.repository;
    } else {
      throw new Error(`Unexpected code generation option "${codeGenerationOption}"`);
    }
  }
  await saveFile({ saveFilePath: projectFilePath, fileContent: JSON.stringify(project, null, 2) });
  electronLog.info('Successfully created project in', projectFolderPath);
  sendEvent({
    eventName: 'dx_rig_create_project',
    eventData: {
      code_used,
      created_extension_in_rig: false,
    }
  });
  return projectFilePath;
}

async function getStatus() {
  clearChildren();
  return {
    isBackendRunning: !!children.backend,
    isFrontendRunning: !!children.frontend,
    port: children.frontend && isHttpServer(children.frontend) ? children.frontend.address().port : undefined,
  };
}

function isHttpServer(child) {
  return child && child.listen;
}

async function stop({ stopOptions }) {
  const backendResult = stopOptions & StopOptions.Backend ? await stopChild(children.backend) : '';
  const frontendResult = stopOptions & StopOptions.Frontend ? await stopChild(children.frontend) : '';
  clearChildren();
  return {
    backendResult,
    frontendResult,
  };

  async function stopChild(child) {
    if (child) {
      if (isHttpServer(child)) {
        // The child is a HTTP server; close it.
        child.close();
        children.frontend = undefined;
        return 'exited';
      }
      if (child.exitCode !== null || child.killed) {
        return 'exited';
      }
      if (process.platform === 'win32') {
        childProcess.spawnSync('taskkill', ['/f', '/pid', child.pid.toString(), '/t']);
      } else {
        child.kill();
      }
      return await new Promise((resolve, _reject) => {
        let hasResolved = false;
        child.on('error', (ex) => {
          hasResolved = hasResolved || (resolve(ex.message), true);
        });
        child.on('exit', (_) => {
          hasResolved = hasResolved || (resolve('exited'), true);
        });
      });
    }
    return 'not running';
  }
}

function clearChildren() {
  ['backend', 'frontend'].forEach((name) => {
    const child = children[name];
    if (!isHttpServer(child) && child && (child.exitCode !== null || child.killed)) {
      children[name] = undefined;
    }
  });
}

function loadFile({ loadFilePath }) {
  return readFile(loadFilePath);
}

async function saveFile({ saveFilePath, fileContent }) {
  if (process.platform === 'win32') {
    fileContent = fileContent.replace(/\n/g, '\r\n').replace(/\r\r/g, '\r');
  }
  await new Promise((resolve, reject) => {
    fs.writeFile(saveFilePath, fileContent, (ex) => {
      if (ex) {
        reject(ex);
      } else {
        resolve();
      }
    });
  });
}

async function fetchExample(repository, projectFolderPath) {
  // Ensure there is a target folder.
  if (!fs.existsSync(projectFolderPath)) {
    fs.mkdirSync(projectFolderPath);
  }
  const exampleFolderName = repository.split('/')[1];
  const exampleFolderPath = resolve(projectFolderPath, exampleFolderName);

  // Determine if git is available.
  const versionResult = childProcess.spawnSync('git', ['--version']);
  if (!versionResult.error && !versionResult.status) {
    // Try to clone it using git.
    const cloneUrl = `https://github.com/${repository}.git`;
    const { exitCode, errorText } = await spawn({ command: 'git', args: ['clone', '--', cloneUrl, exampleFolderPath] });
    if (exitCode) {
      // Failed to clone the repository.
      throw new Error(`git failed with error code ${exitCode}: ${errorText}`);
    }
  } else {
    // Try to fetch the Zip file and unzip it.
    await new Promise((resolve, reject) => {
      const handleError = (ex) => {
        // Failed to fetch or unzip the Zip file.
        if (typeof ex === 'string') {
          ex = new Error(ex);
        }
        reject(ex);
        reject = (_) => { };
      };
      const zipUrl = `https://github.com/${repository}/archive/master.zip`;
      const zipFilePath = join(projectFolderPath, '.master.zip');
      const zipRequest = request(zipUrl);
      const writeStream = fs.createWriteStream(zipFilePath);
      const zipPipe = zipRequest.pipe(writeStream);
      [zipRequest, writeStream, zipPipe].forEach((value) => value.on('error', handleError));
      zipPipe.on('close', () => {
        const readStream = fs.createReadStream(zipFilePath);
        const extractor = unzip.Extract({ path: projectFolderPath });
        const unzipPipe = readStream.pipe(extractor);
        [readStream, extractor, unzipPipe].forEach((value) => value.on('error', handleError));
        unzipPipe.on('close', () => {
          // Successfully fetched and unzipped the Zip file.
          try {
            fs.unlinkSync(zipFilePath);
            fs.renameSync(`${exampleFolderPath}-master`, exampleFolderPath);
            resolve();
          } catch (ex) {
            handleError(ex);
          }
        });
      });
    });
  }
}

function parseCommandLine(commandLine) {
  let quote = '';
  const arg = [], args = [];
  commandLine.split('').forEach((ch) => {
    if (quote === '') {
      if (ch === ' ') {
        args.push(arg.join(''));
        arg.length = 0;
      } else if (ch === '"' || ch === "'") {
        quote = ch;
      } else {
        arg.push(ch);
      }
    } else if (ch === quote) {
      quote = '';
    } else {
      arg.push(ch);
    }
  });
  if (arg.length) {
    args.push(arg.join(''));
  }
  return args;
}

async function spawn({ command, args, options, redirections }) {
  const child = childProcess.spawn(command, args, options);
  const { exitCode, errorText } = await new Promise((resolve, reject) => {
    let errorText = '', hasResolved = false, timerId;
    child.stderr.on('data', (data) => process.stderr.write(data.toString()));
    child.stdout.on('data', (data) => process.stdout.write(data.toString()));
    if (redirections) {
      child.stderr.on('data', (data) => redirections.onStderr(data.toString()));
      child.stdout.on('data', (data) => redirections.onStdout(data.toString()));
    } else {
      child.stderr.on('data', (data) => { errorText += data.toString(); });
    }
    child.on('error', reject);
    child.on('exit', (exitCode) => {
      hasResolved = hasResolved || (clearTimeout(timerId), resolve({ exitCode, errorText }), true);
    });
    if (redirections) {
      timerId = setTimeout(() => {
        hasResolved = hasResolved || (resolve({}), true);
      }, 999);
    }
  });
  return { child, exitCode, errorText };
}
