const app = require('express')();
const { shell } = require('electron');
const electronLog = require('electron-log');
const { basename, join } = require('path');
const request = require('request');
const { format, parse } = require('url');
const uuid = require('uuid');
const api = require('./api');
const { readFile } = require('./async');
const { isDebuggingProduction } = require('./debug');
const { getPort } = require('./project');

if (isDebuggingProduction()) {
  debugger;
}

module.exports = async () => {
  let proxyUrl;
  const frameUrl = process.env.DEV_URL || 'file:';

  app.use(function(req, res, next) {
    if (req.path === '/v1/stats') {
      res.header('Access-Control-Allow-Origin', '*');
      res.header('Access-Control-Allow-Headers', 'Content-Type, X-Requested-With');
    }
    next();
  });

  app.get('/coordinator.js', async (_req, res) => {
    const replacements = [
      { source: 'https://supervisor.ext-twitch.tv', target: proxyUrl },
      { source: 's?:\\/\\/([\\w-]+\\.)*twitch\\.(tv|tech)(:\\d+)?\\/.*$', target: '' },
      { source: 'https://client-event-reporter.twitch.tv', target: proxyUrl },
    ];
    const extensionCoordinatorPath = join(__dirname, 'extension-coordinator.umd.js');
    let body = await readFile(extensionCoordinatorPath);
    body = replacements.reduce((body, replacement) => body.replace(replacement.source, replacement.target), body);
    res.setHeader('Content-Type', 'application/javascript');
    res.writeHead(200);
    res.end(body);
  });

  app.get('/player', async (req, res) => {
    const channel = req.query['channel'];
    const height = req.query['height'];
    const body = `<!DOCTYPE html>
      <html lang="en">
        <body style="height:${height}px;margin:0;">
          <iframe src="https://player.twitch.tv/?parent=localhost&channel=${channel}" frameBorder="0" width="100%" height="100%" scrolling="no" title="${channel}" />
        </body>
      </html>`;
    res.setHeader('Content-Type', 'text/html');
    res.writeHead(200);
    res.end(body);
  });

  app.get('/supervisor/v1/supervisor.css', (_req, res) => getResource('https://supervisor.ext-twitch.tv/supervisor/v1/supervisor.css', [
    { name: 'Content-Security-Policy', value: `frame-ancestors ${frameUrl} ${proxyUrl};` },
    { name: 'Content-Type', value: "text/css" },
  ], [], res));

  app.get('/supervisor/v1/index.html', (_req, res) => getResource('https://supervisor.ext-twitch.tv/supervisor/v1/index.html', [
    { name: 'Content-Security-Policy', value: `frame-ancestors ${frameUrl} ${proxyUrl};` },
    { name: 'Content-Type', value: 'text/html; charset=utf-8' },
  ], [], res));

  app.get('/supervisor/v1/supervisor.js', (_req, res) => getResource('https://supervisor.ext-twitch.tv/supervisor/v1/supervisor.js', [
    { name: 'Content-Security-Policy', value: `frame-ancestors ${frameUrl} ${proxyUrl};` },
    { name: 'Content-Type', value: 'application/javascript' },
  ], [
      { source: 's:\\/\\/(?:[^\\.]+\\.)*?twitch\\.(tv|tech)', target: '|file' },
    ], res));

  app.post('/v1/stats', (_req, res) => {
    res.writeHead(204);
    res.end();
  });

  const browsers = {};
  const keys = {};
  api.add('/browser', async ({ projectFilePath, secret, viewId }) => {
    const keyPath = join(projectFilePath, viewId);
    let key = keys[keyPath];
    if (!key) {
      keys[keyPath] = key = uuid.v1();
    }
    browsers[key] = { projectFilePath, secret, viewId };
    const name = basename(projectFilePath, '.json');
    const url = format({
      ...parse(proxyUrl),
      pathname: '/index.html',
      query: { name, ref: key },
    });
    shell.openExternal(url);
  });

  app.get('/extension-frame.:ext', (req, res) => sendFile(`extension-frame.${req.params.ext}`, res));
  app.get('/static/:directory/:fileName', (req, res) => sendFile(join('static', req.params.directory, req.params.fileName), res));
  app.get('/favicon.ico', (_req, res) => sendFile('favicon.ico', res));
  app.get('/rig-console.js', (_req, res) => sendFile('rig-console.js', res));
  app.get('/index.html', (_req, res) => sendFile('index.html', res));

  app.get('/project/:referenceId', async (req, res) => {
    try {
      const browser = browsers[req.params.referenceId];
      if (!browser) {
        throw new Error(`Invalid reference ${req.params.referenceId}`);
      }
      const { projectFilePath, secret, viewId } = browser;
      const content = await readFile(projectFilePath);
      const project = JSON.parse(content);
      const view = project.extensionViews.filter((view) => view.id === viewId)[0];
      if (!view) {
        throw new Error('Cannot find that extension view.  Did you delete it?');
      }
      res.json({ project, secret, view });
    } catch (ex) {
      res.status(404).json({ message: ex.message });
    }
    res.end();
  });

  app.get('/status', async (_req, res) => {
    const port = await getPort();
    res.json({ port });
  });

  return new Promise((resolve, reject) => {
    const server = app.listen(0, () => {
      const address = server.address();
      proxyUrl = `http://localhost:${address.port}`;
      electronLog.info(`Serving internals from http://${address.address}:${address.port}`);
      resolve(proxyUrl);
    });
    server.on('error', reject);
  });
}

function getResource(url, headers, replacements, res) {
  request(url, (error, response, body) => {
    if (error || response.statusCode !== 200) {
      console.error('Retrieval of Twitch resource', url, 'failed:');
      console.error(error);
      console.error(body);
      res.writeHead(error ? 500 : response.statusCode);
      body = body || JSON.stringify(error);
    } else {
      headers.forEach((header) => res.setHeader(header.name, header.value));
      res.writeHead(200);
      body = replacements.reduce((body, replacement) => body.replace(replacement.source, replacement.target), body);
    }
    res.end(body);
  });
}

function sendFile(fileName, res) {
  const path = join(__dirname, '..', 'build', fileName);
  res.sendFile(path);
}
