import * as React from 'react';
import './component.sass';
import { Dialog } from '../dialog';
import { CoreText, FormGroup, Radio, CheckBox } from 'twitch-core-ui';
import { CertificateExceptionRequest, CertificateExceptionScope, CertificateException } from '../core/models/certificate-exception';
import { LocalStorage } from '../util/local-storage';

interface Props {
  exceptionRequest: CertificateExceptionRequest;
  closeHandler: (certificateException: CertificateException) => void;
}

enum Allowance {
  None = ' None',
  Yes = ' Yes',
  No = ' No',
}

interface State {
  allowance: Allowance;
  isEphemeral: boolean;
  scope?: CertificateExceptionScope;
  [key: string]: Allowance | CertificateExceptionScope | boolean | undefined;
}

const localStorage = new LocalStorage();

export class CheckCertificateDialog extends React.Component<Props, State>{
  public state: State = {
    allowance: Allowance.None,
    isEphemeral: localStorage.isEphemeral,
  };

  componentDidMount() {
    console.warn('Invalid certificate:', this.props.exceptionRequest);
  }

  private onChange = (input: React.FormEvent<HTMLInputElement>) => {
    const { checked, name, value } = input.currentTarget;
    switch(name) {
      case 'allowance':
        this.setState({ allowance: value as Allowance });
        break;
      case 'isEphemeral':
        this.setState({ isEphemeral: !checked });
        localStorage.isEphemeral = !checked;
        break;
      case 'scope':
        this.setState({ scope: value.split(':')[0] as CertificateExceptionScope });
        break;
      default:
        console.error(`unexpected name "${name}" in CheckCertificateDialog.onChange`);
    }
  }

  private close = () => {
    const { allowance, isEphemeral, scope } = this.state;
    const isOrigin = scope === CertificateExceptionScope.Origin;
    const url = new URL(this.props.exceptionRequest.url);
    const value = scope === CertificateExceptionScope.Url ? url.href : url.origin;
    const certificateException = { isAllowed: allowance === Allowance.Yes, isEphemeral, isOrigin, value };
    this.props.closeHandler(certificateException);
  }

  public render() {
    const { exceptionRequest } = this.props;
    const { allowance, isEphemeral, scope } = this.state;
    const url = new URL(exceptionRequest.url);
    return <>
      <Dialog className="check-certificate-dialog" canConfirm={!!scope && allowance !== Allowance.None} confirmText="OK" title="Invalid Certificate" confirmHandler={this.close}>
        <FormGroup hint="See the developer console for more information." label="This resource has an invalid certificate">
          <CoreText>{url.href}</CoreText>
        </FormGroup>
        <FormGroup label="Allow">
          <Radio name="allowance" label={Allowance.Yes} value={Allowance.Yes} onChange={this.onChange} checked={allowance === Allowance.Yes} />
          <Radio name="allowance" label={Allowance.No} value={Allowance.No} onChange={this.onChange} checked={allowance === Allowance.No} />
        </FormGroup>
        <FormGroup label="Scope">
          <Radio name="scope" label={CertificateExceptionScope.Origin} value={CertificateExceptionScope.Origin + ':  ' + url.origin} onChange={this.onChange} checked={CertificateExceptionScope.Origin === scope} />
          <Radio name="scope" label={CertificateExceptionScope.Url} value={CertificateExceptionScope.Url} onChange={this.onChange} checked={CertificateExceptionScope.Url === scope} />
        </FormGroup>
        <CheckBox name="isEphemeral" checked={!isEphemeral} onChange={this.onChange} label="Store in project file" />
      </Dialog>
    </>;
  }
}
