import * as React from 'react';
import { mount } from 'enzyme';
import { ConfigurationServiceView } from './component';
import { createExtensionManifestForTest } from '../tests/constants/extension';
import { DeveloperRigUserId, RigContext } from '../core/models/rig';
import { ChangeEventFn, timeout } from '../tests/utils';

function mockApiFunctions() {
  return {
    ...require.requireActual('../util/api'),
    fetchChannelConfigurationSegments: jest.fn().mockImplementation(() => Promise.resolve({})),
    saveConfigurationSegment: jest.fn(),
  };
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');
function mockIdFunctions() {
  return {
    ...require.requireActual('../util/id'),
    fetchIdForUser: jest.fn().mockImplementation((_, id) => id === 'developerrig' ?
      Promise.resolve(DeveloperRigUserId) : Promise.reject(new Error(`Cannot fetch user "${id}"`))),
  };
}
jest.mock('../util/id', () => mockIdFunctions());
const { fetchIdForUser } = require.requireMock('../util/id');

const rigProject = {
  allowHttpBackend: true,
  backendCommand: 'test',
  backendFolderName: 'test',
  certificateExceptions: [],
  extensionViews: [],
  frontendCommand: 'test',
  frontendFolderName: 'test',
  manifest: createExtensionManifestForTest(),
  name: 'test',
  projectFolderPath: 'test',
  secret: 'test',
  usingRandomFrontendHostingPort: false,
  version: 1,
};
const rx = /="(tw-)?[0-9a-z]{32}"/g;

function createWrapper(isConfigurationHosted: boolean = false) {
  const wrapper = mount(
    <RigContext.Provider value={{ isConfigurationHosted, secret: 'secret', userId: 'userId' }}>
      <ConfigurationServiceView authToken='authToken' rigProject={rigProject} />
    </RigContext.Provider>
  );
  return wrapper;
}

describe('<ConfigurationServiceView />', () => {
  it('renders hosted correctly', () => {
    const wrapper = createWrapper(true);
    expect(wrapper.debug().replace(rx, `="$1${'0'.repeat(32)}"`)).toMatchSnapshot();
    const coreLinkProps = wrapper.find('CoreLink').first().props() as any;
    expect(coreLinkProps.linkTo).toBe('https://dev.twitch.tv/docs/extensions/building/#configuration-service');
  });

  it('renders non-hosted correctly', () => {
    const wrapper = createWrapper(false);
    expect(wrapper.debug().replace(rx, `="$1${'0'.repeat(32)}"`)).toMatchSnapshot();
    const coreLinkProps = wrapper.find('CoreLink').first().props() as any;
    const { id, version } = rigProject.manifest;
    expect(coreLinkProps.linkTo).toBe(`https://dev.twitch.tv/console/extensions/${id}/${version}/capabilities`);
  });

  describe('fetchChannelConfiguration', () => {
    it('fetches by name', async () => {
      const wrapper = createWrapper();
      [
        ['select', 'configurationType', 'developer'],
        ['input[name="channelId"]', 'channelId', 'developerrig'],
      ].forEach(([selector, name, value]) => {
        wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
      });
      wrapper.find('button[aria-label="ConfigurationServiceView:Fetch Configuration"]').simulate('click');
      await timeout();
      expect(fetchIdForUser).toHaveBeenCalled();
      expect(api.fetchChannelConfigurationSegments).toHaveBeenCalled();
      const instance = wrapper.instance() as ConfigurationServiceView;
      expect(instance.state.fetchStatus).toBe('');
    });
  });

  it('fails on unknown name', async () => {
    const wrapper = createWrapper();
    const value = 'unknown';
    [
      ['select', 'configurationType', 'developer'],
      ['input[name="channelId"]', 'channelId', value],
    ].forEach(([selector, name, value]) => {
      wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
    });
      wrapper.find('button[aria-label="ConfigurationServiceView:Fetch Configuration"]').simulate('click');
    await timeout();
    expect(fetchIdForUser).toHaveBeenCalled();
    const instance = wrapper.instance() as ConfigurationServiceView;
    expect(instance.state.fetchStatus).toBe(`Cannot fetch user "${value}"`);
  });

  it('invokes content type change handler', () => {
    const wrapper = createWrapper();
    const instance = wrapper.instance() as ConfigurationServiceView;
    const [name, value] = ['configurationType', 'global'];
    wrapper.find('select').prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
    expect(instance.state[name]).toBe(value);
  });

  describe('save', () => {
    it('invokes save handler for global segment', () => {
      const wrapper = createWrapper();
      [
        ['select', 'configurationType', 'global'],
        ['textarea', 'content', '{}'],
        ['input[name="version"]', 'version', '0.1'],
      ].forEach(([selector, name, value]) => {
        wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
      });
      wrapper.find('button[aria-label="ConfigurationServiceView:Save"]').simulate('click');
      expect(api.saveConfigurationSegment).toHaveBeenCalled();
    });

    it('fails if no channel for non-global segment', () => {
      const wrapper = createWrapper();
      wrapper.find('input[name="channelId"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name: 'channelId', value: '' } });
      wrapper.find('button[aria-label="ConfigurationServiceView:Save"]').simulate('click');
      api.saveConfigurationSegment.mockClear();
      expect(api.saveConfigurationSegment).not.toHaveBeenCalled();
    });

    it('fails on unknown name', async () => {
      const wrapper = createWrapper();
      const instance = wrapper.instance() as ConfigurationServiceView;
      const value = 'unknown';
      [
        ['select', 'configurationType', 'broadcaster'],
        ['input[name="channelId"]', 'channelId', value],
        ['textarea', 'content', '{}'],
        ['input[name="version"]', 'version', '0.1'],
      ].forEach(([selector, name, value]) => {
        wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
      });
      wrapper.find('button[aria-label="ConfigurationServiceView:Save"]').simulate('click');
      await timeout();
      expect(instance.state.fetchStatus).toBe(`Cannot fetch user "${value}"`);
    });
  });

  describe('cancel', () => {
    it('invokes for channel', () => {
      const wrapper = createWrapper();
      const instance = wrapper.instance() as ConfigurationServiceView;
      [
        ['select', 'configurationType', 'broadcaster'],
        ['textarea', 'content', '{}'],
      ].forEach(([selector, name, value]) => {
        wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
      });
      wrapper.find('button[aria-label="ConfigurationServiceView:Reset"]').simulate('click');
      expect(instance.state.content).toBe('');
    });

    it('invokes for global', () => {
      const content = '{}';
      const wrapper = createWrapper();
      [
        ['select', 'configurationType', 'global'],
        ['textarea', 'content', content],
      ].forEach(([selector, name, value]) => {
        wrapper.find(selector).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value } });
      });
      expect(wrapper.state('content')).toBe(content);
      wrapper.find('button[aria-label="ConfigurationServiceView:Reset"]').simulate('click');
      expect(wrapper.state('content')).toBe('');
    });
  });
});
