import * as React from 'react';
import './component.sass';
import { RigContext, RigProject } from '../core/models/rig';
import { fetchChannelConfigurationSegments, saveConfigurationSegment, fetchGlobalConfigurationSegment } from '../util/api';
import { ChannelIdOrNameInput } from '../channel-id-or-name-input';
import { fetchIdForUser } from '../util/id';
import { NavItem } from '../rig-nav';
import { LocalStorage } from '../util/local-storage';
import { Layout, CoreText, TextType, StyledLayout, BorderRadius, FormGroup, Select, Button, SVGAsset, ButtonType, TextArea, Input, InputType, Background, Display, Color, CoreLink } from 'twitch-core-ui';
import { Segment } from 'extension-coordinator';

interface Props {
  authToken: string;
  rigProject: RigProject,
}

enum ConfigurationType {
  Broadcaster = 'broadcaster',
  Developer = 'developer',
  Global = 'global',
}

interface State {
  channelId: string;
  configurationType: ConfigurationType;
  content: string;
  fetchStatus: string;
  lastContent: string;
  lastVersion: string;
  version: string;
  [key: string]: ConfigurationType | string;
}

const localStorage = new LocalStorage();

export class ConfigurationServiceView extends React.Component<Props, State>{
  static contextType = RigContext;
  context!: React.ContextType<typeof RigContext>;
  public state: State = {
    channelId: localStorage.rigLogin!.id,
    configurationType: ConfigurationType.Broadcaster,
    content: '',
    fetchStatus: '',
    lastContent: '',
    lastVersion: '',
    version: '',
  };
  private globalSegment?: Segment;

  async componentDidMount() {
    const { secret, userId } = this.context;
    const { rigProject: { manifest: { id: clientId } } } = this.props;
    this.globalSegment = await fetchGlobalConfigurationSegment(clientId, userId, secret);
    if(this.state.configurationType === ConfigurationType.Global) {
      const { content = '', version = '' } = this.globalSegment || {};
      this.setState({ content, version });
    }
  }

  private onChange = (input: React.FormEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = input.currentTarget;
    this.setState({ [name]: value });
  }

  private onChangeConfigurationType = (input: React.FormEvent<HTMLSelectElement>) => {
    const { name, value } = input.currentTarget;
    this.setState({ [name]: value, lastContent: '', lastVersion: '' });
    if (value === ConfigurationType.Global) {
      const { content = '', version = '' } = this.globalSegment || {};
      this.setState({ content, version });
    } else {
      this.setState({ content: '', version: '' });
    }
  }

  private fetchChannelConfiguration = async () => {
    let channelId = this.state.channelId.trim();
    if (channelId) {
      this.setState({ fetchStatus: 'fetching...' });
      try {
        const { secret, userId } = this.context;
        const { authToken, rigProject: { manifest: { id: clientId } } } = this.props;
        channelId = await fetchIdForUser(authToken, channelId);
        const segmentMap = await fetchChannelConfigurationSegments(clientId, userId, channelId, secret);
        const segment = this.state.configurationType === ConfigurationType.Broadcaster ?
          segmentMap.broadcaster : segmentMap.developer;
        const { content = '', version = '' } = segment || {};
        this.setState({ content, fetchStatus: '', lastContent: content, lastVersion: version, version });
      } catch (ex) {
        this.setState({ fetchStatus: ex.message });
      }
    }
  }

  private canSave = (): boolean => {
    const { channelId, configurationType } = this.state;
    if (configurationType !== ConfigurationType.Global && !channelId.trim()) {
      // The non-global configuration segments need a channel ID.
      return false;
    }
    return true;
  }

  private save = async () => {
    if (this.canSave()) {
      const { content, configurationType, version } = this.state;
      let { channelId } = this.state;
      try {
        this.setState({ fetchStatus: 'saving...' });
        if (configurationType !== ConfigurationType.Global) {
          channelId = await fetchIdForUser(this.props.authToken, channelId);
        }
        const { secret, userId } = this.context;
        const { rigProject: { manifest: { id: clientId } } } = this.props;
        await saveConfigurationSegment(clientId, userId, secret, configurationType,
          channelId, content.trim(), version.trim());
        this.setState({ fetchStatus: '' });
      } catch (ex) {
        this.setState({ fetchStatus: ex.message });
      }
    }
  }

  private reset = () => {
    const { configurationType, lastContent, lastVersion } = this.state;
    const { content = '', version = '' } = configurationType === ConfigurationType.Global ?
      this.globalSegment || {} :
      { content: lastContent, version: lastVersion };
    this.setState({ content, version });
  }

  public render() {
    const configurationServiceUrl = 'https://dev.twitch.tv/docs/extensions/building/#configuration-service';
    const { manifest } = this.props.rigProject;
    const capabilitiesUrl = `https://dev.twitch.tv/console/extensions/${manifest.id}/${manifest.version}/capabilities`;

    return (
      <StyledLayout background={Background.Alt} fullHeight padding={3}>
        <CoreText bold color={Color.Base} type={TextType.H3}>{NavItem.ConfigurationService}</CoreText>
        <StyledLayout background={Background.Base} borderLeft borderRight borderTop borderRadius={{ topLeft: BorderRadius.Medium, topRight: BorderRadius.Medium }} margin={{ top: 2 }} padding={2}>
          <div>
            {this.context.isConfigurationHosted ? <>
              <CoreText className="configuration-service-view__text">The configuration service helps you build your extension by
                removing the burden of writing a back end to store persistent channel- and extension-specific data.  It then
                provides this data on extension load, eliminating the need for your back end to handle traffic from end users for
                this scenario.  This tool enables you to easily access and set that configuration data during development or
                production.</CoreText>
            </> : <>
              <CoreText className="configuration-service-view__text">This extension
                is <CoreText type={TextType.Strong}>not</CoreText> currently using the configuration service.  Visit
                the <CoreLink ariaLabel="ConfigurationServiceView:Capabilities" linkTo={capabilitiesUrl}>Capabilities</CoreLink> section
                of this version of your extension to learn more.  If you have already enabled using the configuration service,
                refresh your Extension Manifest by clicking the "Refresh Manifest" button in the {NavItem.ProjectOverview} tab.</CoreText>
              </>}
            <CoreLink ariaLabel="ConfigurationServiceView:Learn more about Extension Configuration Service" linkTo={configurationServiceUrl}>Learn more about Extension Configuration Service</CoreLink>.
          </div>
          <Layout margin={{ top: 1 }}>
            <FormGroup label="Configuration Type">
              <Select name="configurationType" value={this.state.configurationType} onChange={this.onChangeConfigurationType}>
                <option value={ConfigurationType.Broadcaster}>Broadcaster</option>
                <option value={ConfigurationType.Developer}>Developer</option>
                <option value={ConfigurationType.Global}>Global</option>
              </Select>
            </FormGroup>
          </Layout>
          {this.state.configurationType !== ConfigurationType.Global && <>
            <Layout margin={{ top: 1 }}>
              <FormGroup label="Channel ID or Name">
                <Layout display={Display.InlineBlock}>
                  <ChannelIdOrNameInput name="channelId" value={this.state.channelId} onChange={this.onChange} />
                </Layout>
              </FormGroup>
              <Button ariaLabel="ConfigurationServiceView:Fetch Configuration" icon={SVGAsset.Refresh} type={ButtonType.Text} onClick={this.fetchChannelConfiguration}>Fetch Configuration</Button>
              <CoreText color={Color.Base} type={TextType.Span}>{this.state.fetchStatus}</CoreText>
            </Layout>
          </>}
          <Layout margin={{ top: 1 }}>
            <FormGroup label="Configuration">
              <TextArea name="content" rows={5} value={this.state.content} onChange={this.onChange} />
            </FormGroup>
          </Layout>
          <Layout margin={{ top: 1 }}>
            <FormGroup label="Version">
              <Input name="version" type={InputType.Text} value={this.state.version} onChange={this.onChange} />
            </FormGroup>
          </Layout>
        </StyledLayout>
        <StyledLayout background={Background.Alt2} borderBottom borderLeft borderRight borderRadius={{ bottomLeft: BorderRadius.Medium, bottomRight: BorderRadius.Medium }} padding={{ bottom: 1, top: 1 }}>
          <Layout display={Display.Inline} margin={{ left: 2 }}>
            <Button ariaLabel="ConfigurationServiceView:Reset" type={ButtonType.Hollow} onClick={this.reset}>Reset</Button>
          </Layout>
          <Layout display={Display.Inline} margin={{ left: 1 }}>
            <Button ariaLabel="ConfigurationServiceView:Save" onClick={this.save}>Save</Button>
          </Layout>
        </StyledLayout>
      </StyledLayout>
    );
  }
}
